#include "l10n.h"

#include <maps/wikimap/mapspro/libs/masstransit/masstransit.h>
#include "stop.h"

#include <maps/libs/locale/include/codes.h>
#include <maps/libs/locale/include/convert.h>

#include <algorithm>

namespace maps::wiki::masstransit {

namespace {

const size_t NAME_TYPE_OFFICIAL = 0;
const size_t NAME_TYPE_RENDER = 1;

bool hasRenderName(const std::vector<Name>& names, locale::Lang lang)
{
    return std::find_if(names.begin(), names.end(),
        [&](const Name& name) {
            return name.lang == lang && name.nameType == NAME_TYPE_RENDER;
        }) != names.end();
}

void removeOfficialName(std::vector<Name>& names, locale::Lang lang)
{
    auto end = std::remove_if(names.begin(), names.end(),
        [&](const Name& name) {
            return name.lang == lang && name.nameType == NAME_TYPE_OFFICIAL;
        });
    names.erase(end, names.end());
}

} // namespace

Name::Name()
    : lang(locale::Lang::Unknown)
    , isLocal(false)
    , nameType(0)
{ }

YmapsdfLocalizationHandler::YmapsdfLocalizationHandler(Masstransit& masstransit)
    : YmapsdfObjectHandler(masstransit, masstransit.localizations)
{ }

std::vector<FtType>
YmapsdfLocalizationHandler::ftTypes() const
{
    return {
        FtType::TransportBusStop,
        FtType::TransportMetroStation,
        FtType::TransportMetroExit,
        FtType::TransportWaterwayWharf,
        FtType::TransportWaterwaySeaport,
        FtType::TransportWaterwayRiverport,

        FtType::TransportBusRoute,
        FtType::TransportTrolleybusRoute,
        FtType::TransportTramRoute,
        FtType::TransportMinibusRoute,
        FtType::TransportMetroLine,
        FtType::TransportWaterwayRoute,
        FtType::TransportMetroCableLine,
        FtType::TransportMetroTramLine,
        FtType::TransportMetroBusLine,
        FtType::TransportMetroFunicularLine,

        FtType::TransportBusThread,
        FtType::TransportTramThread,
        FtType::TransportMetroThread,
        FtType::TransportWaterwayThread
    };
}

std::string
YmapsdfLocalizationHandler::loadIdsSql() const
{
    return "SELECT nm_id AS id FROM ft_nm JOIN ft USING(ft_id) "
           "WHERE ft_type_id IN " + toString(ftTypes()) + " "
           "AND name_type IN (0,1);";
}

std::string
YmapsdfLocalizationHandler::loadRowsSqlTemplate() const
{
    return
        "SELECT "
            "ft_nm.nm_id, "
            "ft_nm.ft_id AS id, "
            "ft.ft_type_id, "
            "ft_nm.lang, "
            "ft_nm.is_local, "
            "ft_nm.name_type, "
            "ft_nm.name "
        "FROM ft_nm "
        "JOIN ft USING (ft_id) "
        "WHERE nm_id IN %1%;";
}

void
YmapsdfLocalizationHandler::addObject(const pqxx::row& tuple)
{
    const auto ftType = getAttr<FtType>(tuple, ymapsdf::FT_TYPE_ID);
    if (ftType == FtType::TransportMetroExit) {
        return; // no names for exits
    }

    const auto objectId = getAttr<DBID>(tuple, ymapsdf::ID);

    Name nm;
    nm.lang = getAttr<locale::Language>(tuple, ymapsdf::LANG).lang;
    nm.isLocal = getAttr<bool>(tuple, ymapsdf::IS_LOCAL);
    nm.nameType = getAttr<size_t>(tuple, ymapsdf::NAME_TYPE);
    nm.name = getAttr<std::string>(tuple, ymapsdf::NAME);

    if (!masstransit_.localizations.count(objectId)) {
        auto l10n = std::make_shared<Localization>();
        l10n->objectId = objectId;
        l10n->mtrType = mtrType(ftType);
        insert(objectId, l10n);
    }

    auto& localization = masstransit_.localizations[objectId];

    if (hasRenderName(localization.names, nm.lang)) {
        return;
    }

    if (nm.nameType == NAME_TYPE_RENDER) {
        removeOfficialName(localization.names, nm.lang);
    }

    localization.names.emplace_back(std::move(nm));
}

void
YmapsdfLocalizationHandler::update()
{
    for (const auto& object : masstransit_.stops()) {
        const Stop& exit = dynamic_cast<const Stop&>(*object);
        if (exit.ftType == FtType::TransportMetroExit) {
            replaceExitNamesWithStationNames(exit.stopId);
        }
    }
}

void
YmapsdfLocalizationHandler::replaceExitNamesWithStationNames(DBID exitId)
{
    const auto& exit = masstransit_.stops[exitId];
    const auto stationId = exit.parentStopId;

    REQUIRE(masstransit_.localizations.count(stationId),
        "Station " << stationId << " has no names");

    if (!masstransit_.localizations.count(exitId)) {
        auto l10n = std::make_shared<Localization>();
        l10n->objectId = exitId;
        l10n->mtrType = mtrType(exit.ftType);
        masstransit_.localizations.insert(exitId, l10n);
    }

    auto& exitNames = masstransit_.localizations[exitId];
    const auto& stationNames = masstransit_.localizations[stationId];

    exitNames.names = stationNames.names;
}

MtrLocalizationHandler::MtrLocalizationHandler(Masstransit& masstransit)
    : MtrObjectHandler(masstransit, masstransit.localizations)
{ }

void
MtrLocalizationHandler::writeObject(StreamMap& ostreams, const Object& object)
{
    const auto& l10n = dynamic_cast<const Localization&>(object);
    const auto objectMtrId = idMap().getMtrId(l10n.objectId);
    const auto mtrType = l10n.mtrType;

    auto& ostream = ostreams[fileNames().front()];
    for (const auto& nm : l10n.names) {
        ostream
            << toString(mtrType) << FIELD_SEPARATOR
            << objectMtrId << FIELD_SEPARATOR
            << nm.lang << FIELD_SEPARATOR
            /*no grammar*/ << FIELD_SEPARATOR
            << nm.name << std::endl;

        ostream
            << toString(mtrType) << FIELD_SEPARATOR
            << objectMtrId << FIELD_SEPARATOR
            << nm.lang << FIELD_SEPARATOR
            << STR_NOMINATIVE << FIELD_SEPARATOR
            << nm.name << std::endl;
    }
}

} // namespace maps::wiki::masstransit
