#include "thread.h"

#include <maps/wikimap/mapspro/libs/masstransit/masstransit.h>
#include "route.h"
#include "thread_closure.h"

#include <maps/libs/enum_io/include/enum_io.h>

namespace maps::wiki::masstransit {

namespace {

constexpr enum_io::Representations<ThreadType> THREAD_TYPE_ENUM_REPRESENTATION {
    {ThreadType::Linear, "linear"},
    {ThreadType::Circular, "circular"}
};

void
addRouteForStops(Masstransit& masstransit, DBID threadId, DBID routeId)
{
    const auto& threadStops = masstransit.threadStops[threadId];
    for (const auto& threadStop : threadStops.stopsInThread) {
        masstransit.stops[threadStop.stopId].routeIds.insert(routeId);
    }
}

void
copySchedule(Masstransit& masstransit, DBID threadId, const common::Schedule& schedule)
{
    auto threadCalendar = std::make_shared<Calendar>();
    threadCalendar->threadId = threadId;
    threadCalendar->mtrSchedules.push_back(schedule);

    masstransit.calendars.insert(threadId, threadCalendar);
}

} // namespace

DEFINE_ENUM_IO(ThreadType, THREAD_TYPE_ENUM_REPRESENTATION);

YmapsdfThreadHandler::YmapsdfThreadHandler(Masstransit& masstransit)
    : YmapsdfObjectHandler(masstransit, masstransit.threads)
{ }

std::vector<FtType>
YmapsdfThreadHandler::ftTypes() const
{
    return {
        FtType::TransportBusThread,
        FtType::TransportTramThread,
        FtType::TransportWaterwayThread,
        FtType::TransportMetroThread
    };
}

StringVector
YmapsdfThreadHandler::attrs() const
{
    return {
        attr::TYPE,
        attr::CLOSURE
    };
}

StringVector
YmapsdfThreadHandler::rolesToMasters() const
{
    return {role::ASSIGNED_THREAD};
}

void
YmapsdfThreadHandler::addObject(const pqxx::row& tuple)
{
    auto thread = std::make_shared<Thread>();

    const auto threadId = getAttr<DBID>(tuple, ymapsdf::ID);

    thread->threadId = threadId;
    thread->routeId = getAttr<DBID>(tuple, role::ASSIGNED_THREAD);
    thread->type = getAttr<ThreadType>(tuple, attr::TYPE);
    thread->name = getAttr<std::string>(tuple, ymapsdf::NAME, "");
    thread->ftType = getAttr<FtType>(tuple, ymapsdf::FT_TYPE_ID);

    auto mtrId = getAttr<std::string>(tuple, ymapsdf::SOURCE_ID, std::to_string(thread->threadId));
    idMap().add(thread->threadId, mtrId, MtrType::Thread);

    insert(threadId, thread);
    if (getAttr<bool>(tuple, attr::CLOSURE, false)) {
        masstransit_.threadClosures.insert(threadId,
            std::make_shared<ThreadClosure>(threadId));
    }
}

void
YmapsdfThreadHandler::updateObject(Object& object)
{
    const auto& thread = dynamic_cast<const Thread&>(object);
    const auto threadId = thread.threadId;

    try {
        addRouteForStops(masstransit_, threadId, thread.routeId);

        const auto& route = masstransit_.routes[thread.routeId];
        const auto& transportOperator = masstransit_.operators[route.operatorId];

        if (!masstransit_.calendars.count(threadId)) {
            copySchedule(masstransit_, threadId, transportOperator.schedule);
        }
    } catch (std::exception& e) {
        DATA_ERROR() << "Thread id=" << threadId << " error: " << e.what();
        throw;
    }
}

MtrThreadHandler::MtrThreadHandler(Masstransit& masstransit)
    : MtrObjectHandler(masstransit, masstransit.threads)
{ }

void
MtrThreadHandler::writeObject(StreamMap& ostreams, const Object& object)
{
    const auto& thread = dynamic_cast<const Thread&>(object);
    auto& ostream = ostreams[fileNames().front()];

    ostream
        << idMap().getMtrId(thread.threadId) << FIELD_SEPARATOR
        << idMap().getMtrId(thread.routeId) << FIELD_SEPARATOR
        << thread.type << FIELD_SEPARATOR
        << /*ignore backThreadId*/ FIELD_SEPARATOR
        << thread.name << std::endl;
}

} // namespace maps::wiki::masstransit
