#include "thread_stops.h"

#include <maps/wikimap/mapspro/libs/masstransit/masstransit.h>

namespace maps::wiki::masstransit {

namespace {

DBID
addOrderedStopAndReturnId(
    std::vector<ThreadStop>& stopsInThread,
    std::vector<TravelTimeDatum>& travelTimeData,
    ThreadStop&& threadStop,
    TravelTimeDatum&& travelTimeDatum)
{
    const auto threadStopId = threadStop.threadStopId;
    const auto seqId = stopsInThread.size();

    threadStop.seqId = seqId;
    travelTimeDatum.seqId = seqId;

    size_t previousDepartureTimeSec = seqId ? travelTimeData.back().departureTimeSec : 0;
    travelTimeDatum.arrivalTimeSec += previousDepartureTimeSec; // travelTime -> arrivalTime
    travelTimeDatum.departureTimeSec += travelTimeDatum.arrivalTimeSec; // waitTime -> departureTime

    stopsInThread.emplace_back(std::move(threadStop));
    travelTimeData.emplace_back(std::move(travelTimeDatum));
    return threadStopId;
}

void
addStop(
    ThreadStops& threadStops,
    std::vector<TravelTimeDatum>& travelTimeData,
    ThreadStop&& threadStop,
    TravelTimeDatum&& travelTimeDatum,
    DBID previousThreadStopId)
{
    auto& stopsInThread = threadStops.stopsInThread;
    auto& previousIdToThreadStop = threadStops.mtr.previousIdToThreadStop;
    auto& previousIdToTravelTime = threadStops.mtr.previousIdToTravelTime;

    DBID lastStopInThreadId = stopsInThread.empty() ? 0 : stopsInThread.back().threadStopId;

    if (previousThreadStopId != lastStopInThreadId) {
        previousIdToThreadStop.insert({previousThreadStopId, threadStop});
        previousIdToTravelTime.insert({previousThreadStopId, travelTimeDatum});
    } else {
        lastStopInThreadId = addOrderedStopAndReturnId(
            stopsInThread,
            travelTimeData,
            std::move(threadStop),
            std::move(travelTimeDatum));
        for (auto iterThreadStop = previousIdToThreadStop.find(lastStopInThreadId);
            iterThreadStop != previousIdToThreadStop.end();
            iterThreadStop = previousIdToThreadStop.find(lastStopInThreadId))
        {
            auto iterTravelTime = previousIdToTravelTime.find(lastStopInThreadId);
            lastStopInThreadId = addOrderedStopAndReturnId(
                stopsInThread,
                travelTimeData,
                std::move(iterThreadStop->second),
                std::move(iterTravelTime->second));
            previousIdToThreadStop.erase(iterThreadStop);
            previousIdToTravelTime.erase(iterTravelTime);
        }
    }
}

void
setFlagsOnStart(ThreadStopFlags& flags)
{
    flags.important = true;
    flags.noDropOff = true;
}

void
setFlagsOnEnd(ThreadStopFlags& flags)
{
    flags.important = true;
    flags.noBoarding = true;
}

} // namespace

ThreadStopFlags::ThreadStopFlags()
    : noBoarding(false)
    , noDropOff(false)
    , important(false)
{ }

ThreadStopFlags::ThreadStopFlags(const pqxx::row& tuple)
    : noBoarding(getAttr<bool>(tuple, attr::NO_EMBARKATION, false))
    , noDropOff(getAttr<bool>(tuple, attr::NO_DISEMBARKATION, false))
    , important(getAttr<bool>(tuple, attr::IS_TERMINAL, false))
{ }

std::ostream&
operator<<(std::ostream& ostream, const ThreadStopFlags& flags)
{
    bool noFlags = true;
    if (flags.noBoarding) {
        if (!noFlags) {
            ostream << FLAG_SEPARATOR;
        }
        noFlags = false;
        ostream << "nb";
    }
    if (flags.important) {
        if (!noFlags) {
            ostream << FLAG_SEPARATOR;
        }
        noFlags = false;
        ostream << "imp";
    }
    if (flags.noDropOff) {
        if (!noFlags) {
            ostream << FLAG_SEPARATOR;
        }
        noFlags = false;
        ostream << "nd";
    }

    return ostream;
}

bool
isWaypoint(const ThreadStopFlags& flags)
{
    return flags.noBoarding && flags.noDropOff;
}

ThreadStop::ThreadStop()
    : threadStopId(0)
    , stopId(0)
    , seqId(0)
    , flags()
{ }

YmapsdfThreadStopsHandler::YmapsdfThreadStopsHandler(Masstransit& masstransit)
    : YmapsdfObjectHandler(masstransit, masstransit.threadStops)
{ }

std::vector<FtType>
YmapsdfThreadStopsHandler::ftTypes() const
{
    return {FtType::TransportThreadStop};
}

StringVector
YmapsdfThreadStopsHandler::attrs() const
{
    return {
        attr::TRAVEL_TIME,
        attr::WAIT_TIME,
        attr::NO_EMBARKATION,
        attr::NO_DISEMBARKATION,
        attr::IS_TERMINAL
    };
}

StringVector
YmapsdfThreadStopsHandler::rolesToMasters() const
{
    return {role::ASSIGNED_THREAD_STOP, role::PART};
}

StringVector
YmapsdfThreadStopsHandler::rolesToSlaves() const
{
    return {role::PREVIOUS};
}

void
YmapsdfThreadStopsHandler::addObject(const pqxx::row& tuple)
{
    const auto threadId = getAttr<DBID>(tuple, role::PART);
    const auto threadStopId = getAttr<DBID>(tuple, ymapsdf::ID);
    const auto previousThreadStopId = getAttr<DBID>(tuple, role::PREVIOUS, 0);

    const auto travelTimeMtrId = getTravelTimeMtrId(threadId);
    const auto travelTimeId = idMap().getDBID(travelTimeMtrId, MtrType::TravelTime);

    ThreadStop threadStop;
    threadStop.threadStopId = threadStopId;
    threadStop.seqId = 0;
    threadStop.stopId = getAttr<DBID>(tuple, role::ASSIGNED_THREAD_STOP);
    threadStop.flags = ThreadStopFlags(tuple);

    TravelTimeDatum travelTimeDatum;
    travelTimeDatum.seqId = 0;
    travelTimeDatum.arrivalTimeSec = getAttr<size_t>(tuple, attr::TRAVEL_TIME, 0);
    travelTimeDatum.departureTimeSec = getAttr<size_t>(tuple, attr::WAIT_TIME, 0);

    if (!masstransit_.threadStops.count(threadId)) {
        auto threadStops = std::make_shared<ThreadStops>();
        threadStops->threadId = threadId;
        insert(threadId, threadStops);
    }
    if (!masstransit_.travelTimes.count(travelTimeId)) {
        auto travelTime = std::make_shared<TravelTime>(travelTimeId, threadId);
        masstransit_.travelTimes.insert(travelTimeId, travelTime);
    }

    addStop(
        masstransit_.threadStops[threadId],
        masstransit_.travelTimes[travelTimeId].data,
        std::move(threadStop),
        std::move(travelTimeDatum),
        previousThreadStopId);
}

MtrThreadStopsHandler::MtrThreadStopsHandler(Masstransit& masstransit)
    : MtrObjectHandler(masstransit, masstransit.threadStops)
{ }

void
MtrThreadStopsHandler::writeObject(StreamMap& ostreams, const Object& object)
{
    const auto& threadStops = dynamic_cast<const ThreadStops&>(object);
    if (!masstransit_.threads.count(threadStops.threadId)) {
        return;
    }
    auto threadMtrId = idMap().getMtrId(threadStops.threadId);

    size_t seqId = 0;
    for (const auto& threadStop : threadStops.stopsInThread) {
        const auto& stop = masstransit_.stops[threadStop.stopId];
        if (isWaypoint(threadStop.flags) || stop.isWaypoint) {
            continue;
        }

        auto flags = threadStop.flags;
        if (threadStop.seqId == 0) {
            setFlagsOnStart(flags);
        } else if (threadStop.seqId == threadStops.stopsInThread.size() - 1) {
            setFlagsOnEnd(flags);
        }

        auto& ostream = ostreams[fileNames().front()];
        ostream
            << threadMtrId << FIELD_SEPARATOR
            << seqId << FIELD_SEPARATOR
            << idMap().getMtrId(threadStop.stopId) << FIELD_SEPARATOR
            << flags << std::endl;
        ++seqId;
    }
}

} // namespace maps::wiki::masstransit
