#include "transitions.h"

#include <maps/wikimap/mapspro/libs/masstransit/masstransit.h>

namespace maps::wiki::masstransit {

namespace {

void
writeTransition(
    std::ostream& ostream,
    const std::string& srcStopMtrId,
    const std::string& dstStopMtrId,
    bool blocked,
    size_t timeSec)
{
    ostream
        << srcStopMtrId << FIELD_SEPARATOR
        << dstStopMtrId << FIELD_SEPARATOR;
    if (blocked) {
        ostream << "INF";
    } else {
        ostream << timeSec;
    }
    ostream << std::endl;
}

} // namespace

YmapsdfTransitionHandler::YmapsdfTransitionHandler(Masstransit& masstransit)
    : YmapsdfObjectHandler(masstransit, masstransit.transitions)
{ }

std::vector<FtType>
YmapsdfTransitionHandler::ftTypes() const
{
    return {FtType::TransportMetroTransition, FtType::TransportMetroPassageway};
}

StringVector
YmapsdfTransitionHandler::attrs() const
{
    return {attr::TIME, attr::ONEWAY, attr::IS_VIRTUAL};
}

StringVector
YmapsdfTransitionHandler::rolesToSlaves() const
{
    return {role::STATION_A, role::STATION_B, role::STATION, role::EXIT};
}

void
YmapsdfTransitionHandler::addObject(const pqxx::row& tuple)
{
    auto transition = std::make_shared<Transition>();

    transition->transitionId = getAttr<DBID>(tuple, ymapsdf::ID);
    const auto ftType = getAttr<FtType>(tuple, ymapsdf::FT_TYPE_ID);
    transition->isVirtual = getAttr<bool>(tuple, attr::IS_VIRTUAL);
    if (ftType == FtType::TransportMetroPassageway) {
        transition->srcStopId = getAttr<DBID>(tuple, role::STATION);
        transition->dstStopId = getAttr<DBID>(tuple, role::EXIT);
        if (!transition->isVirtual) {
            transition->direction = reverse(getAttr<Direction>(tuple, attr::ONEWAY));
        }
    } else {
        ASSERT(ftType == FtType::TransportMetroTransition);
        transition->srcStopId = getAttr<DBID>(tuple, role::STATION_A);
        transition->dstStopId = getAttr<DBID>(tuple, role::STATION_B);
        if (!transition->isVirtual) {
            transition->direction = getAttr<bool>(tuple, attr::ONEWAY)
                ? Direction::Forward
                : Direction::Both;
        }
    }

    if (!transition->isVirtual) {
        const auto strTime = getAttr<std::string>(tuple, attr::TIME);
        if (strTime == "INF") {
            transition->blocked = true;
            transition->timeSec = 0;
        } else {
            transition->blocked = false;
            transition->timeSec = std::stoul(strTime);
        }
    }
    insert(transition->transitionId, transition);
}

MtrTransitionHandler::MtrTransitionHandler(Masstransit& masstransit)
    : MtrObjectHandler(masstransit, masstransit.transitions)
{ }

void
MtrTransitionHandler::writeObject(StreamMap& ostreams, const Object& object)
{
    const auto& transition = dynamic_cast<const Transition&>(object);
    if (transition.isVirtual ||
        masstransit_.stops[transition.srcStopId].isWaypoint ||
        masstransit_.stops[transition.dstStopId].isWaypoint)
    {
        return;
    }

    const auto srcStopMtrId = idMap().getMtrId(transition.srcStopId);
    const auto dstStopMtrId = idMap().getMtrId(transition.dstStopId);

    auto& ostream = ostreams[fileNames().front()];
    if (transition.direction != Direction::Backward) {
        writeTransition(ostream, srcStopMtrId, dstStopMtrId, *transition.blocked, *transition.timeSec);
    }
    if (transition.direction != Direction::Forward) {
        writeTransition(ostream, dstStopMtrId, srcStopMtrId, *transition.blocked, *transition.timeSec);
    }
}

} // namespace maps::wiki::masstransit
