#include <yandex/maps/wiki/masstransit/convert.h>
#include <yandex/maps/wiki/masstransit/geobase.h>

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/pgpool/include/pgpool3.h>
#include <maps/libs/common/include/exception.h>

using namespace std::chrono_literals;

namespace {

const unsigned MASTER_SIZE = 1;
const unsigned MASTER_MAX_SIZE = 1;
const unsigned SLAVE_SIZE = 6;
const unsigned SLAVE_MAX_SIZE = 15;

const auto GET_TIMEOUT_MS = 0s;
const auto PING_INTERVAL_MS = 5s;
const auto PING_TIMEOUT_MS = 30s;

maps::pgpool3::Pool createPgPool(const std::string& connStr)
{
    auto poolConstants = maps::pgpool3::PoolConstants(
        MASTER_SIZE, MASTER_MAX_SIZE, SLAVE_SIZE, SLAVE_MAX_SIZE);

    poolConstants.getTimeoutMs = GET_TIMEOUT_MS;
    poolConstants.pingIntervalMs = PING_INTERVAL_MS;
    poolConstants.pingTimeoutMs = PING_TIMEOUT_MS;

    return maps::pgpool3::Pool(
        connStr,
        std::move(poolConstants));
}

} // namespace

int main(int argc, char* argv[]) try
{
    maps::cmdline::Parser parser;

    auto connStr = parser.string("conn").required()
        .help("connection string");
    auto schema = parser.string("schema").required()
        .help("schema name");
    auto geodataPath = parser.string("geodata-path").required()
        .help("geodata6.bin directory");
    auto mtrPath = parser.string("mtr-path").required()
        .help("resulting mtr directory");
    auto dataErrLog = parser.string("data-error-log").required()
        .help("data error log file");

    parser.parse(argc, argv);

    maps::wiki::masstransit::setGeodataBinPath(geodataPath);

    auto ymapsdfPool = createPgPool(connStr);
    maps::wiki::masstransit::ymapsdf2mtr(
        ymapsdfPool,
        schema,
        mtrPath,
        dataErrLog);

    return EXIT_SUCCESS;
} catch (const maps::Exception& e) {
    ERROR() << "Conversion failed: " << e;
    return EXIT_FAILURE;
} catch (const std::exception& e) {
    ERROR() << "Conversion failed: " << e.what();
    return EXIT_FAILURE;
}
