#include "magic_strings.h"
#include <yandex/maps/wiki/mds_dataset/exception.h>
#include <yandex/maps/wiki/mds_dataset/export_metadata.h>
#include <yandex/maps/wiki/common/string_utils.h>

#include <ostream>

namespace maps {
namespace wiki {
namespace mds_dataset {

class ExportMetadata::Impl
{
public:
    Impl(BasicMetadata basicMetadata, Subset subset, IsTested tested)
        : basic_(std::move(basicMetadata))
        , subset_(subset)
        , tested_(tested)
    {}

    std::tuple<const BasicMetadata&, const Subset&, const IsTested&> tie() const
    {
        return std::tie(basic_, subset_, tested_);
    }

    BasicMetadata basic_;
    Subset subset_;
    IsTested tested_;
};

ExportMetadata::ExportMetadata(BasicMetadata basicMetadata, Subset subset, IsTested tested)
    : impl_(new Impl(std::move(basicMetadata), subset, tested))
{}

COPYABLE_PIMPL_DEFINITIONS(ExportMetadata)

const BasicMetadata& ExportMetadata::basic() const
{
    return impl_->basic_;
}

Subset ExportMetadata::subset() const
{
    return impl_->subset_;
}

IsTested ExportMetadata::tested() const
{
    return impl_->tested_;
}

const DatasetID& ExportMetadata::id() const
{
    return basic().id();
}

const Region& ExportMetadata::region() const
{
    return basic().region();
}


template <>
void validateMetadata(const ExportMetadata& metadata)
{
    if (metadata.id().empty()) {
        throw InvalidMetadata() << "Empty export dataset id";
    }

    auto status = metadata.basic().status();
    if (status > DatasetStatus::Max) {
        throw InvalidMetadata() << "Invalid export dataset status value: "
                << static_cast<int>(status);
    }

    auto subset = metadata.subset();
    if (subset > Subset::Max) {
        throw InvalidMetadata() << "Invalid export dataset subset value: "
                << static_cast<int>(subset);
    }
}

bool ExportMetadata::operator<(const ExportMetadata& other) const
{
    return impl_->tie() < other.impl_->tie();
}

bool ExportMetadata::operator==(const ExportMetadata& other) const
{
    return impl_->tie() == other.impl_->tie();
}

std::ostream& operator<<(std::ostream& out, IsTested tested)
{
    return out << static_cast<int>(tested);
}

std::ostream& operator<<(std::ostream& out, const ExportMetadata& metadata)
{
    return out << "{ basicMetadata: " << metadata.basic()
               << ", subset: " << metadata.subset()
               << ", tested: " << metadata.tested()
               << " }";
}

} // mds_dataset
} // wiki
} // maps
