#include "sql.h"
#include "magic_strings.h"
#include "include/yandex/maps/wiki/mds_dataset/export_metadata.h"
#include <yandex/maps/wiki/common/pg_utils.h>
#include <yandex/maps/wiki/common/string_utils.h>

#include <sstream>
#include <iostream>

namespace maps {
namespace wiki {
namespace mds_dataset {
namespace sql {

typedef pqxx::transaction_base Txn;

namespace {

std::string makeFileLinkRow(const DatasetID& id, const Region& region, const FileLink& fileLink, Txn& txn)
{
    std::ostringstream os;
    os << "(" << txn.quote(id)
       << ", " << txn.quote(region)
       << ", " << txn.quote(fileLink.mdsKey().groupId)
       << ", " << txn.quote(fileLink.mdsKey().path)
       << ", " << txn.quote(fileLink.name())
       << ", " << txn.quote(fileLink.readingUrl())
       << ")";
    return os.str();
}

const std::string TABLE_FILES_COLUMNS = common::join(
    std::vector<std::string>{
        col::DATASET_ID,
        col::REGION,
        col::MDS_GROUP_ID,
        col::MDS_PATH,
        col::NAME,
        col::URL
    },
    ","
);

} // namespace

template<> const std::string
Traits<ExportMetadata>::TABLE_META = tbl::EXPORT_META;

template<> const std::string
Traits<ExportMetadata>::TABLE_FILES = tbl::EXPORT_FILES;

template<> const std::string
Traits<ExportMetadata>::META_COLUMNS = common::join(
    std::vector<std::string>{
        col::ID,
        col::REGION,
        col::STATUS,
        col::CREATED,
        col::SUBSET,
        col::TESTED
    },
    ","
);


template <>
void writeMetadata(const ExportMetadata& metadata, Txn& txn)
{
    std::ostringstream query;
    query << "INSERT INTO " << tbl::EXPORT_META
            << " (" << Traits<ExportMetadata>::META_COLUMNS << ")"
        << " VALUES"
            << " (" << txn.quote(metadata.basic().id())
            << ", '" << metadata.region() << "'"
            << ", " <<  static_cast<int>(metadata.basic().status())
            << ", " << func::TO_TIMESTAMP << "("
                    << timestampToSec(metadata.basic().created()) << ")"
            << ", " << static_cast<int>(metadata.subset())
            << ", " << (metadata.tested() == IsTested::Yes ? "TRUE" : "FALSE")
            << ");";
    txn.exec(query.str());
}

template <typename MetaData>
void setDatasetStatus(
    const DatasetID& id,
    const Region& region,
    DatasetStatus status,
    Txn& txn)
{
    std::ostringstream query;
    query << "UPDATE " << Traits<MetaData>::TABLE_META << "\n"
          << "SET " << col::STATUS << " = " << static_cast<int>(status) << "\n"
          << "WHERE " << col::ID << " = " << txn.quote(id) << " "
          << "AND " << col::REGION << " = " << txn.quote(region) << ";";
    txn.exec(query.str());
}

template <typename MetaData>
void writeFileLink(
    const DatasetID& id,
    const Region& region,
    const FileLink& fileLink,
    Txn& txn)
{
    std::ostringstream query;

    query << "INSERT INTO " << Traits<MetaData>::TABLE_FILES
            << " (" << TABLE_FILES_COLUMNS << ")"
            << " VALUES " << makeFileLinkRow(id, region, fileLink, txn) << ";";
    txn.exec(query.str());
}

template <typename MetaData>
void writeFileLinks(
    const DatasetID& id,
    const Region& region,
    const FileLinks& fileLinks,
    Txn& txn)
{
    if (fileLinks.empty()) {
        return;
    }

    std::ostringstream query;
    using namespace std::placeholders;
    auto rowMaker = std::bind(makeFileLinkRow, id, region, _1, std::ref(txn));

    query << "INSERT INTO " << Traits<MetaData>::TABLE_FILES
            << " (" << TABLE_FILES_COLUMNS << ")"
            << " VALUES " << common::join(fileLinks, rowMaker, ",")
            << ";";
    txn.exec(query.str());
}

template <typename MetaData>
FileLinks loadFileLinks(
        const DatasetID& id,
        const Region& region,
        Txn& txn)
{
    std::ostringstream query;

    query << "SELECT " << TABLE_FILES_COLUMNS << "\n"
          << "FROM " << Traits<MetaData>::TABLE_FILES << "\n"
          << "WHERE " << col::DATASET_ID << " = " << txn.quote(id) << "\n"
          << "AND " << col::REGION << " = " << txn.quote(region) << ";";

    FileLinks fileLinks;
    auto rows = txn.exec(query.str());
    for (const auto& row : rows) {
        mds::Key mdsKey{row[col::MDS_GROUP_ID].as<std::string>(),
                        row[col::MDS_PATH].as<std::string>()};
        auto name = row[col::NAME].as<std::string>();
        auto url = row[col::URL].as<std::string>();

        fileLinks.emplace_back(std::move(mdsKey), std::move(name), std::move(url));
    }
    return fileLinks;
}

template <>
std::vector<Dataset<ExportMetadata>> loadDatasets(
        Txn& txn,
        const std::string& filterClause,
        const std::string& limitClause)
{
    std::ostringstream query;
    query << "SELECT "
            << col::ID
            << ", " << tbl::EXPORT_META << "." << col::REGION << "\n"
            << ", " << col::STATUS
            << ", " << col::SUBSET
            << ", " << col::TESTED << "\n"
            << ", " << func::EXTRACT << "("
                << "EPOCH FROM " << col::CREATED << ") AS " << col::CREATED << "\n"
            << ", array_agg(" << col::MDS_GROUP_ID << ") AS " << col::MDS_GROUP_ID << "\n"
            << ", array_agg(" << col::MDS_PATH << ") AS " << col::MDS_PATH << "\n"
            << ", array_agg(" << col::NAME << ") AS " << col::NAME << "\n"
            << ", array_agg(" << col::URL << ") AS " << col::URL << "\n"
        << "FROM " << tbl::EXPORT_META << "\n"
        << "LEFT JOIN " << tbl::EXPORT_FILES << " ON ( "
            << tbl::EXPORT_META << "." << col::REGION << " = "
                <<  tbl::EXPORT_FILES << "." << col::REGION
            << " AND " <<  col::ID << " = " <<  col::DATASET_ID << ")\n"
        << filterClause << "\n"
        << "GROUP BY "
            << col::ID
            << ", " << tbl::EXPORT_META << "." << col::REGION
            << ", " << col::STATUS
            << ", " << col::SUBSET
            << ", " << col::TESTED
            << ", " << col::CREATED << "\n"
        << "ORDER BY " << sql::col::CREATED << " DESC\n"
        << limitClause;

    std::vector<Dataset<ExportMetadata>> datasets;
    auto rows = txn.exec(query.str());
    for (const auto& row : rows) {
        auto id = row[col::ID].as<DatasetID>();
        auto region = row[col::REGION].as<std::string>(NO_REGION);
        auto status = static_cast<DatasetStatus>(row[col::STATUS].as<int>());
        auto ts = row[col::CREATED].as<int64_t>();
        auto created = Timestamp{std::chrono::seconds{ts}};
        auto subset = static_cast<Subset>(row[col::SUBSET].as<int>());
        auto tested = row[col::TESTED].as<bool>() ? IsTested::Yes : IsTested::No;
        auto mdsGroupIds = common::parseSqlArray(row[col::MDS_GROUP_ID].as<std::string>());
        auto mdsPaths = common::parseSqlArray(row[col::MDS_PATH].as<std::string>());
        auto names = common::parseSqlArray(row[col::NAME].as<std::string>());
        auto urls = common::parseSqlArray(row[col::URL].as<std::string>());
        ASSERT(mdsGroupIds.size() == urls.size());
        ASSERT(mdsPaths.size() == urls.size());
        ASSERT(names.size() == urls.size());

        ExportMetadata metadata(BasicMetadata(std::move(id), std::move(region), status, created), subset, tested);
        FileLinks fileLinks;
        for (size_t i = 0; i < urls.size(); ++i) {
            mds::Key mdsKey{mdsGroupIds[i], mdsPaths[i]};
            auto& name = names[i];
            auto& url = urls[i];
            fileLinks.emplace_back(std::move(mdsKey), std::move(name), std::move(url));
        }
        datasets.push_back({std::move(metadata), std::move(fileLinks)});
    }
    return datasets;
}

template <typename MetaData>
void deleteFileLinks(const DatasetID& id, const Region& region, Txn& txn)
{
    std::ostringstream query;
    query << "DELETE FROM " << Traits<MetaData>::TABLE_FILES << "\n"
          << "WHERE " << col::DATASET_ID << " = " << txn.quote(id) << " "
          << "AND " << col::REGION << " = '" << region << "';";
    txn.exec(query.str());
}

// Explicit instantiation
#define INSTANTIATE_FUNCTIONS(MetaData)               \
template void setDatasetStatus<MetaData>(             \
        const DatasetID& id,                          \
        const Region& region,                         \
        DatasetStatus status,                         \
        Txn& txn);                                    \
                                                      \
template void writeFileLink<MetaData>(                \
        const DatasetID& id,                          \
        const Region& region,                         \
        const FileLink& fileLink,                     \
        Txn& txn);                                    \
                                                      \
template void writeFileLinks<MetaData>(               \
        const DatasetID& id,                          \
        const Region& region,                         \
        const FileLinks& fileLinks,                   \
        Txn& txn);                                    \
                                                      \
template FileLinks loadFileLinks<MetaData>(           \
        const DatasetID& id,                          \
        const Region& region,                         \
        Txn& txn);                                    \
                                                      \
template void deleteFileLinks<MetaData>(              \
        const DatasetID& id,                          \
        const Region& region,                         \
        Txn& txn);                                    \

INSTANTIATE_FUNCTIONS(ExportMetadata)

} // sql
} // mds_dataset
} // wiki
} // maps
