#include "../include/point_to_bld.h"

#include <yandex/maps/wiki/common/geom_utils.h>
#include <yandex/maps/wiki/revision/common.h>
#include <maps/wikimap/mapspro/libs/views/include/query_builder.h>

#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/distance.h>

namespace maps::wiki::misc {

namespace {

const double DEFAULT_AZIMUTH = 0.0;

} // unnamed namespace

std::optional<PointToBldResult> pullPointTowardsBld(
    pqxx::transaction_base& txn,
    revision::DBID branchId,
    const geolib3::Point2& mercPoint,
    bool findInsideBld,
    double searchRadius,
    const std::string& bldNumber)
{
    const auto ratio = geolib3::MercatorRatio::fromMercatorPoint(mercPoint);

    views::QueryBuilder qb(branchId);

    qb.selectFields(
        "ST_AsBinary(ST_ClosestPoint(ST_Boundary(bld.the_geom), ST_GeomFromWKB($1, 3395))) AS point, "
        "ST_Distance(ST_GeomFromWKB($1, 3395), ST_Boundary(bld.the_geom)) as distance, "
        "degrees(ST_Azimuth(ST_ClosestPoint(ST_Boundary(bld.the_geom), ST_GeomFromWKB($1, 3395)), ST_GeomFromWKB($1, 3395))) as azimuth, "
        "bld.id as bld_id");

    qb.fromTable(views::TABLE_OBJECTS_A, "bld");
    if (!bldNumber.empty()) {
        qb.fromTable(views::TABLE_OBJECTS_P, "addr");
    }

    std::string whereClause =
        " ST_DWithin(ST_GeomFromWKB($1, 3395), bld.the_geom, $2)"
        " AND bld.domain_attrs ? 'cat:bld'"
        " AND " +
            std::string(findInsideBld ? "" : "NOT ") +
            "ST_Contains(bld.the_geom, ST_GeomFromWKB($1, 3395))";
    if (!bldNumber.empty()) {
        whereClause +=
            " AND addr.domain_attrs ? 'cat:addr'"
            " AND LOWER(addr.service_attrs -> 'srv:render_label') = LOWER($3)"
            " AND ST_Contains(bld.the_geom, addr.the_geom)";
    }
    qb.whereClause(std::move(whereClause));

    auto query = qb.query() +
        " ORDER BY distance " + (findInsideBld ? "DESC" : "ASC") +
        " LIMIT 1";

    pqxx::result rows;

    if (!bldNumber.empty()) {
        rows = txn.exec_params(
            query,
            pqxx::binarystring{geolib3::WKB::toString(mercPoint)},
            ratio.toMercatorUnits(searchRadius),
            bldNumber
        );
    } else {
        rows = txn.exec_params(
            query,
            pqxx::binarystring{geolib3::WKB::toString(mercPoint)},
            ratio.toMercatorUnits(searchRadius)
        );
    }

    ASSERT(rows.size() <= 1);
    if (rows.empty()) {
        return std::nullopt;
    }

    PointToBldResult result;
    const auto& row = *rows.begin();

    auto wkb = pqxx::binarystring(row["point"]).str();
    result.point = geolib3::WKB::read<geolib3::Point2>(wkb);
    result.distance = ratio.toMeters(row["distance"].as<double>());
    result.azimuth =
        row["azimuth"].is_null()
        ? DEFAULT_AZIMUTH
        : row["azimuth"].as<double>();
    result.isInsideBld = findInsideBld;
    result.bldId = row["bld_id"].as<revision::DBID>();

    if (result.isInsideBld) {
        result.azimuth += 180.0;
        if (result.azimuth >= 360.0) {
            result.azimuth -= 360.0;
        }
    }
    return result;
}

} // namespace maps::wiki::misc
