#include <yandex/maps/wiki/outsource/task_result.h>

#include <yandex/maps/wiki/common/date_time.h>
#include <yandex/maps/wiki/common/geom.h>
#include <yandex/maps/wiki/common/revision_utils.h>
#include <yandex/maps/wiki/diffalert/revision/aoi_diff_loader.h>
#include <yandex/maps/wiki/outsource/config.h>
#include <yandex/maps/wiki/revision/branch_manager.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>
#include <yandex/maps/wiki/threadutils/threadpool.h>

#include <maps/libs/log8/include/log8.h>

#include <algorithm>

namespace maps {
namespace wiki {
namespace outsource {

namespace rev = revision;
namespace rf = rev::filters;
namespace da = diffalert;

namespace {

const double DEFAULT_MIN_LINEAR_OBJECT_INTERSECTION_RATIO = 0.85;
const std::vector<std::pair<std::string, double>> MIN_LINEAR_OBJECT_INTERSECTION_RATIOS =
{
    { "rd", 0.05 },
    { "rd_el", 0.5 }
};

const std::string CAT_OUTSOURCE_REGION = "cat:outsource_region";

const std::string ATTR_STATUS = "outsource_region:status";
const std::string ATTR_NAME = "outsource_region:name";

const std::string STATUS_CAN_START = "can_start";
const std::string STATUS_AWAITING_CHECK = "awaiting_check";
const std::string STATUS_AWAITING_CORRECTIONS = "awaiting_corrections";
const std::string STATUS_AWAITING_BILLING = "awaiting_billing";

const std::string ERR_INTERNAL_ERROR = "Внутренняя ошибка";
const std::string ERR_INCOMPLETE_STATUS_SEQUENCE = "Последовательность статусов не завершена";

rev::Revisions
loadRegionRevisionsByFilter(
    const rev::DBID& branchId,
    pqxx::transaction_base& txn)
{
    rev::RevisionsGateway rg(txn, rev::BranchManager(txn).load(branchId));
    auto snapshot = rg.stableSnapshot(rg.headCommitId());
    return snapshot.objectRevisionsByFilter(
            rf::ObjRevAttr::isNotDeleted()
            && rf::ObjRevAttr::isNotRelation()
            && rf::Geom::defined()
            && rf::Attr(CAT_OUTSOURCE_REGION).defined()
            && rf::Attr(ATTR_STATUS) == STATUS_AWAITING_BILLING);
}

rev::Revisions
loadRegionRevisionsByIds(
    const rev::DBIDSet& ids,
    const rev::DBID& branchId,
    pqxx::transaction_base& txn)
{
    rev::RevisionsGateway rg(txn, rev::BranchManager(txn).load(branchId));
    auto snapshot = rg.stableSnapshot(rg.headCommitId());
    auto revisionsById = snapshot.objectRevisions(ids);
    rev::Revisions revisions;
    for (const auto& id : ids) {
        auto revIt = revisionsById.find(id);
        REQUIRE(revIt != revisionsById.end(), "could not find object id: " << id);
        revisions.push_back(std::move(revIt->second));
    }
    return revisions;
}

struct Interval
{
    std::string startedAt;
    rev::DBID startCommitId;
    std::string endedAt;
    rev::DBID endCommitId;
};

struct CompletedTask
{
    outsource::FixedTaskAttributes fixedTaskAttrs;
    outsource::VaryingTaskAttributes varyingTaskAttrs;
    Interval workInterval;
    std::vector<Interval> correctionIntervals;
};

struct IncompleteTask
{
    outsource::FixedTaskAttributes fixedTaskAttrs;
    outsource::VaryingTaskAttributes varyingTaskAttrs;
    std::string startedAt;
    rev::DBID startCommitId;
};

struct AttrChange
{
    rev::DBID commitId;
    std::string changedAt;
    std::string status;
    outsource::FixedTaskAttributes fixedTaskAttrs;
    outsource::VaryingTaskAttributes varyingTaskAttrs;
};

/// Expected status sequence:
/// can_start ->
/// awaiting_check ->
/// (zero or more times: awaiting_corrections -> awaiting_check) ->
/// awaiting_billing
void calcCompletedTasks(
    const std::vector<AttrChange>& attrChanges,
    RegionCalcPolicy calcPolicy,
    std::vector<CompletedTask>& completedTasks,
    boost::optional<IncompleteTask>& incompleteTask)
{
    auto currIt = attrChanges.begin();
    auto advanceToFirst = [&](const std::initializer_list<std::string>& statuses)
    {
        currIt = std::find_if(currIt, attrChanges.end(), [&](const AttrChange& attrChange) {
            return std::find(statuses.begin(), statuses.end(), attrChange.status) != statuses.end();
        });
    };
    auto advanceToLastChangeToCanStart = [&]()
    {
        auto foundIt = attrChanges.end();
        while (currIt != attrChanges.end()) {
            if (currIt->status == STATUS_CAN_START
                && (currIt == attrChanges.begin()
                || std::prev(currIt)->status != STATUS_CAN_START)) {
                    foundIt = currIt;
            }
            ++currIt;
        }
        currIt = foundIt;
    };

    while (true) {
        if (calcPolicy == RegionCalcPolicy::AllTasks) {
            advanceToFirst({STATUS_CAN_START});
        } else {
            advanceToLastChangeToCanStart();
        }
        if (currIt == attrChanges.end()) {
            return;
        }

        auto begin = currIt;
        const auto& fixedTaskAttrs = begin->fixedTaskAttrs;
        IncompleteTask maybeIncompleteTask{fixedTaskAttrs, begin->varyingTaskAttrs, begin->changedAt, begin->commitId};

        advanceToFirst({STATUS_AWAITING_CHECK});
        if (currIt == attrChanges.end()) {
            incompleteTask = maybeIncompleteTask;
            return;
        }

        auto end = currIt;

        Interval workInterval{begin->changedAt, begin->commitId, end->changedAt, end->commitId};

        std::vector<Interval> correctionIntervals;
        while (true) {
            advanceToFirst({STATUS_AWAITING_CORRECTIONS, STATUS_AWAITING_BILLING});
            if (currIt == attrChanges.end()) {
                incompleteTask = maybeIncompleteTask;
                return;
            }
            if (currIt->status == STATUS_AWAITING_BILLING) {
                break;
            }

            auto begin = currIt;

            advanceToFirst({STATUS_AWAITING_CHECK});
            if (currIt == attrChanges.end()) {
                incompleteTask = maybeIncompleteTask;
                return;
            }
            auto end = currIt;

            correctionIntervals.push_back(
                Interval{begin->changedAt, begin->commitId, end->changedAt, end->commitId});
        }

        completedTasks.push_back(
            CompletedTask{
                fixedTaskAttrs,
                currIt->varyingTaskAttrs,
                std::move(workInterval),
                std::move(correctionIntervals)});
    }
}

struct Region
{
    rev::DBID id;
    std::string name;
    da::Geom geom;
    std::vector<CompletedTask> completedTasks;
    boost::optional<IncompleteTask> incompleteTask;
};

const std::string& getAttrValue(
    const revision::ObjectRevision& rev,
    const std::string& attrName)
{
    const auto& attrs = *rev.data().attributes;
    auto attrIt = attrs.find(attrName);
    REQUIRE(attrIt != attrs.end(),
            "revision id: " << rev.id() << " without " << attrName << " attribute");
    return attrIt->second;
}

Region loadRegion(
    const rev::ObjectRevision& currentRegionRev,
    RegionCalcPolicy calcPolicy,
    const revision::DBID& branchId,
    pqxx::transaction_base& txn)
{
    auto regionId = currentRegionRev.id().objectId();

    REQUIRE(!currentRegionRev.data().deleted, "object with id: " << regionId << " is deleted");

    REQUIRE(currentRegionRev.data().geometry, "no geometry for object id: " << regionId);
    da::Geom geom(*currentRegionRev.data().geometry);
    REQUIRE(geom.geometryTypeName() == da::Geom::geomTypeNamePolygon,
            "geometry is not polygon but: " << geom.geometryTypeName());

    getAttrValue(currentRegionRev, CAT_OUTSOURCE_REGION); // check category
    const auto& currentStatus = getAttrValue(currentRegionRev, ATTR_STATUS);
    REQUIRE(currentStatus == STATUS_AWAITING_BILLING, "bad current status: " << currentStatus);
    const auto& regionName = getAttrValue(currentRegionRev, ATTR_NAME);

    rev::RevisionsGateway rg(txn, rev::BranchManager(txn).load(branchId));
    auto histSnapshot = rg.historicalSnapshot(currentRegionRev.id().commitId());
    auto revIds = histSnapshot.revisionIdsByFilter(rf::ObjRevAttr::objectId() == regionId);
    auto revisions = histSnapshot.reader().loadRevisions(revIds);
    revisions.sort(
            [](const rev::ObjectRevision& left, const rev::ObjectRevision& right)
            {
                return left.id().commitId() < right.id().commitId();
            });

    rev::DBIDSet commitIds;
    for (const auto& id : revIds) {
        commitIds.insert(id.commitId());
    }
    std::map<rev::DBID, rev::Commit> commitsById;
    for (auto& commit : rev::Commit::load(txn, rf::CommitAttr::id().in(commitIds))) {
        auto commitId = commit.id();
        commitsById.emplace(commitId, std::move(commit));
    }

    std::vector<AttrChange> attrChanges;
    for (const auto& rev : revisions) {
        auto commitIt = commitsById.find(rev.id().commitId());
        REQUIRE(commitIt != commitsById.end(),
                "couldn't load commit for revision id: " << rev.id());
        const auto& commit = commitIt->second;

        const auto& status = getAttrValue(rev, ATTR_STATUS);
        outsource::FixedTaskAttributes fixedTaskAttrs(rev);
        outsource::VaryingTaskAttributes varyingTaskAttrs(rev);

        if (attrChanges.empty()
                || attrChanges.back().status != status
                || attrChanges.back().fixedTaskAttrs != fixedTaskAttrs
                || attrChanges.back().varyingTaskAttrs != varyingTaskAttrs) {
            attrChanges.push_back(
                    AttrChange{
                        commit.id(),
                        commit.createdAt(),
                        status,
                        std::move(fixedTaskAttrs),
                        std::move(varyingTaskAttrs)});
        }
    }

    std::vector<CompletedTask> completedTasks;
    boost::optional<IncompleteTask> incompleteTask;
    calcCompletedTasks(attrChanges, calcPolicy, completedTasks, incompleteTask);

    return Region{
            regionId,
            regionName,
            std::move(geom),
            std::move(completedTasks),
            std::move(incompleteTask)};
}

std::vector<TaskResult> calcRegionTaskResults(
    const rev::ObjectRevision& currentRev,
    RegionCalcPolicy calcPolicy,
    const revision::DBID& branchId,
    pgpool3::Pool& pool,
    const da::EditorConfig& editorConfig,
    const outsource::Config& outsourceConfig)
{
    std::vector<TaskResult> results;

    auto regionId = currentRev.id().objectId();
    try {
        auto txn = common::getReadTransactionForCommit(
                pool, branchId, currentRev.id().commitId(),
                [](const std::string& entry) { INFO() << entry; });

        auto region = loadRegion(currentRev, calcPolicy, branchId, *txn);

        rev::RevisionsGateway rg(*txn, rev::BranchManager(*txn).load(branchId));
        auto calcIntervalDiff = [&](
            const Interval& interval,
            const outsource::FixedTaskAttributes& taskAttrs,
            const outsource::CalcDiff& calcDiff)
            -> double
        {
            da::AoiDiffLoader loader(
                    editorConfig,
                    region.geom,
                    rg.stableSnapshot(interval.startCommitId),
                    rg.stableSnapshot(interval.endCommitId),
                    branchId,
                    branchId,
                    *txn);
            loader.setDefaultMinLinearObjectIntersectionRatio(DEFAULT_MIN_LINEAR_OBJECT_INTERSECTION_RATIO);
            for (const auto& categoryRatioPair : MIN_LINEAR_OBJECT_INTERSECTION_RATIOS) {
                loader.setMinLinearObjectIntersectionRatio(categoryRatioPair.first, categoryRatioPair.second);
            }
            return calcDiff(taskAttrs, loader);
        };

        for (const auto& task : region.completedTasks) {
            TaskResult result(regionId);
            result.regionName = region.name;

            result.workStart = task.workInterval.startedAt;
            result.workEnd = task.workInterval.endedAt;
            if (!task.correctionIntervals.empty()) {
                result.correctionStart = task.correctionIntervals.front().startedAt;
                result.correctionEnd = task.correctionIntervals.back().endedAt;
            }

            try {
                const auto& taskTypeInfo = outsourceConfig.taskTypeInfo(task.fixedTaskAttrs.taskType, task.fixedTaskAttrs.companyId);
                result.taskType = task.fixedTaskAttrs.taskType;
                result.taskTypeLabel = taskTypeInfo.labelRu;
                result.fcSet = task.fixedTaskAttrs.fcSet;
                result.rate = taskTypeInfo.rate;

                double work = calcIntervalDiff(task.workInterval, task.fixedTaskAttrs, taskTypeInfo.calcWork);
                double corrections = 0.0;
                for (const auto& interval : task.correctionIntervals) {
                    corrections += calcIntervalDiff(interval, task.fixedTaskAttrs, taskTypeInfo.calcCorrections);
                }
                const auto& companyInfo = outsourceConfig.companyInfo(task.fixedTaskAttrs.companyId);

                result.companyName = companyInfo.name;
                result.taxIncluded = companyInfo.taxIncluded;
                result.outsourcerLogin = task.varyingTaskAttrs.outsourcerLogin;
                result.quality = task.varyingTaskAttrs.quality;
                result.complexityRate = task.varyingTaskAttrs.complexityRate;
                result.taxRate = companyInfo.taxRate;

                result.workAmount = work;
                result.moneyAmount = work * taskTypeInfo.rate *
                    task.varyingTaskAttrs.complexityRate * companyInfo.taxRate;
                result.correctionAmount = corrections;

                const double EPS = 1e-8;
                if (std::fabs(work) > EPS) {
                    result.correctionPercentage = 100.0 * corrections / work;
                }
            } catch (const maps::Exception& ex) {
                WARN() << "maps::Exception while processing task for region id: " << regionId
                       << " type: " << task.fixedTaskAttrs.taskType
                       << " starting at commit: " << task.workInterval.startCommitId
                       << ": " << ex;
                result.errorDescription = ERR_INTERNAL_ERROR;
            } catch (const std::exception& ex) {
                WARN() << "std::exception while processing task for region id: " << regionId
                       << " type: " << task.fixedTaskAttrs.taskType
                       << " starting at commit: " << task.workInterval.startCommitId
                       << ": " << ex.what();
                result.errorDescription = ERR_INTERNAL_ERROR;
            }

            results.push_back(std::move(result));
        }

        if (region.incompleteTask) {
            TaskResult result(regionId);
            result.taskType = region.incompleteTask->fixedTaskAttrs.taskType;
            result.taskTypeLabel = outsourceConfig.taskTypeInfo(
                region.incompleteTask->fixedTaskAttrs.taskType,
                region.incompleteTask->fixedTaskAttrs.companyId).labelRu;
            result.fcSet = region.incompleteTask->fixedTaskAttrs.fcSet;
            result.companyName = outsourceConfig.companyInfo(
                region.incompleteTask->fixedTaskAttrs.companyId).name;
            result.taxIncluded = outsourceConfig.companyInfo(
                region.incompleteTask->fixedTaskAttrs.companyId).taxIncluded;
            result.outsourcerLogin =
                region.incompleteTask->varyingTaskAttrs.outsourcerLogin;
            result.quality =
                region.incompleteTask->varyingTaskAttrs.quality;
            result.complexityRate =
                region.incompleteTask->varyingTaskAttrs.complexityRate;
            result.taxRate = outsourceConfig.companyInfo(
                region.incompleteTask->fixedTaskAttrs.companyId).taxRate;
            result.workStart = region.incompleteTask->startedAt;
            result.errorDescription = ERR_INCOMPLETE_STATUS_SEQUENCE;
            results.push_back(std::move(result));
        }
    } catch (const maps::Exception& ex) {
        WARN() << "maps::Exception while processing region id " << regionId << ": " << ex;
        TaskResult result(regionId);
        result.errorDescription = ERR_INTERNAL_ERROR;
        results.push_back(std::move(result));
    } catch (const std::exception& ex) {
        WARN() << "std::exception while processing region id " << regionId << ": " << ex.what();
        TaskResult result(regionId);
        result.errorDescription = ERR_INTERNAL_ERROR;
        results.push_back(std::move(result));
    }

    return results;
}

} // namespace

TaskResults calcTaskResults(
        const revision::DBIDSet& regionIds,
        RegionCalcPolicy calcPolicy,
        revision::DBID branchId,
        pgpool3::Pool& pgPool,
        const da::EditorConfig& editorConfig,
        const outsource::Config& outsourceConfig)
{
    rev::Revisions regionRevisions;
    {
        auto txn = pgPool.masterReadOnlyTransaction();
        regionRevisions = regionIds.empty()
            ? loadRegionRevisionsByFilter(branchId, *txn)
            : loadRegionRevisionsByIds(regionIds, branchId, *txn);
    }

    TaskResults taskResults;

    if (regionRevisions.size() == 1) {
        taskResults[regionRevisions.front().id().objectId()] =
            calcRegionTaskResults(regionRevisions.front(), calcPolicy,
                branchId, pgPool, editorConfig, outsourceConfig);
    } else {
        ThreadPool threadPool(pgPool.state().constants.slaveMaxSize);
        for (const auto& rev : regionRevisions) {
            auto& regionResults = taskResults[rev.id().objectId()];
            threadPool.push(
                [&]()
                {
                    regionResults = calcRegionTaskResults(rev, calcPolicy,
                        branchId, pgPool, editorConfig, outsourceConfig);
                });
        }
        threadPool.shutdown();
    }

    return taskResults;
}

} // namespace outsource
} // namespace wiki
} // namespace maps
