#include <maps/wikimap/mapspro/libs/poi_conflicts/include/poi_conflicts.h>

#include <maps/libs/enum_io/include/enum_io.h>
#include <maps/libs/json/include/value.h>
#include <library/cpp/resource/resource.h>
#include <maps/libs/common/include/exception.h>

#include <string>

namespace maps::wiki::poi_conflicts {

namespace {
const size_t MIN_ZOOM_FOR_CONFLICTS = 19;
} // namespace

constexpr enum_io::Representations<ConflictSeverity> CONFLICT_SEVERITY_ENUM_REPRESENTATION {
    {ConflictSeverity::Critical, "critical"},
    {ConflictSeverity::Error, "error"},
};

DEFINE_ENUM_IO(ConflictSeverity, CONFLICT_SEVERITY_ENUM_REPRESENTATION);

PoiConflicts::PoiConflicts()
    : maxDistanceMercator_(0)
{
    static const std::string POI_CONFLICTS_ZOOMS_RESOURCE_ID = "maps_wiki_poi_conflicts_zooms";
    static const std::string GP_VS_GP = "gp_vs_gp";
    const auto resourceJson = json::Value::fromString(NResource::Find(POI_CONFLICTS_ZOOMS_RESOURCE_ID));
    REQUIRE(resourceJson.isObject() && resourceJson.hasField(GP_VS_GP),
        "Config should be json object with gp_vs_gp field.");
    const auto& geoproductVsGeorpoduct = resourceJson[GP_VS_GP];
    REQUIRE(geoproductVsGeorpoduct.isObject(),
        "gp_vs_gp should be object with zoom to distance map");
    for (const auto& field : geoproductVsGeorpoduct.fields()) {
        const auto zoom  = std::stoul(field);
        if (zoom < MIN_ZOOM_FOR_CONFLICTS) {
            continue;
        }
        const auto distance = geoproductVsGeorpoduct[field].as<double>();
        maxDistanceMercator_ = std::max(distance, maxDistanceMercator_);
        zoomToConflictDistanceMercator_.emplace(zoom, distance);
    }
    REQUIRE(verify(),
        "Config distances should increase with decreased zoom.");
}

std::optional<TZoom>
PoiConflicts::conflictZoom(double distanceMercator, Kind /*kind*/) const
{
    if (distanceMercator > maxConflictDistanceMercator()) {
        return {};
    }
    for (auto it = zoomToConflictDistanceMercator_.cbegin();
        it != zoomToConflictDistanceMercator_.cend();
        ++it)
    {
        const auto zoom = it->first;
        auto nextIt = std::next(it);
        if (nextIt == zoomToConflictDistanceMercator_.end()) {
            return zoom;
        }
        const auto dist = it->second;
        const auto nextDist = nextIt->second;
        if (distanceMercator <= dist && distanceMercator > nextDist) {
            return zoom;
        }
    }
    return {};
}

bool
PoiConflicts::verify() const
{
    if (zoomToConflictDistanceMercator_.empty()) {
        return false;
    }
    for (auto it = zoomToConflictDistanceMercator_.cbegin(), nextIt = std::next(it);
         nextIt != zoomToConflictDistanceMercator_.cend();
         ++it, ++nextIt)
    {
        const auto dist = it->second;
        const auto nextDist = nextIt->second;
        if (dist <= nextDist) {
            return false;
        }
    }
    return true;
}

} //namespace maps::wiki::poi_conflicts
