#include <maps/wikimap/mapspro/libs/poi_feed/include/feed_object_data.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/json/include/builder.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/log8/include/log8.h>

using namespace std::string_literals;

namespace maps::wiki::poi_feed {
namespace json {
const auto ID = "id"s;
const auto PERMALINK = "permalink"s;
const auto TO_DELETE = "toDelete"s;
const auto REVISION = "revision"s;
const auto LON = "lon"s;
const auto LAT = "lat"s;
const auto RUBRIC = "rubric"s;
const auto SECONDARY_RUBRICS = "secondaryRubrics"s;
const auto FT_TYPE_ID = "ftTypeId"s;
const auto NAMES = "names"s;
const auto LANG = "lang"s;
const auto NAME = "name"s;
const auto SHORT_NAMES = "shortNames"s;
const auto INDOOR_LEVEL_UNIVERSAL = "indoorLevelUniversal"s;
const auto IS_ADVERT = "IsAdvert"s;
const auto IS_PLATINUM = "IsPlatinum"s;
const auto HAS_OWNER = "HasOwner"s;
const auto INDOOR_PLAN_ID = "indoorPlanId"s;
const auto IS_GEOPRODUCT = "IsGeoproduct"s;
}

namespace {

const double POSITION_COMPARE_TOLERANCE_METERS = 1.0;

} // namespace


FeedObjectData::FeedObjectData(const std::string& jsonString) try {
    auto parsedJson = maps::json::Value::fromString(jsonString);
    nmapsId_ = boost::lexical_cast<ObjectId>(parsedJson[json::ID].as<std::string>());
    permalink_ = boost::lexical_cast<ObjectId>(parsedJson[json::PERMALINK].as<std::string>());
    toDelete_ = parsedJson[json::TO_DELETE].as<bool>();
    actualizationDate_ = chrono::sinceEpochToTimePoint<std::chrono::milliseconds>(
        boost::lexical_cast<size_t>(parsedJson[json::REVISION].as<std::string>()));
    if (parsedJson.hasField(json::INDOOR_LEVEL_UNIVERSAL)) {
        indoorLevelUniversal_ = parsedJson[json::INDOOR_LEVEL_UNIVERSAL].toString();
    }
    if (toDelete_) {
        return;
    }
    if (parsedJson.hasField(json::LON) && parsedJson.hasField(json::LAT)) {
        pos_ = Position {
            parsedJson[json::LON].as<double>(),
            parsedJson[json::LAT].as<double>()
        };
    }
    for (const auto& nameJsonValue : parsedJson[json::NAMES]) {
        names_.insert({
            nameJsonValue[json::LANG].as<std::string>(),
            nameJsonValue[json::NAME].as<std::string>()
        });
    }
    for (const auto& shortNameJsonValue : parsedJson[json::SHORT_NAMES]) {
        shortNames_.insert({
            shortNameJsonValue[json::LANG].as<std::string>(),
            shortNameJsonValue[json::NAME].as<std::string>()
        });
    }
    if (parsedJson.hasField(json::RUBRIC)) {
        rubricId_ = parsedJson[json::RUBRIC].as<RubricId>();
    }
    if (parsedJson.hasField(json::SECONDARY_RUBRICS)) {
        for (const auto& secondaryRubricValue : parsedJson[json::SECONDARY_RUBRICS]) {
            secondaryRubricIds_.insert(secondaryRubricValue.as<RubricId>());
        }
    }
    if (parsedJson.hasField(json::FT_TYPE_ID)) {
        ftTypeId_ = parsedJson[json::FT_TYPE_ID].as<FtTypeId>();
    }
    if (parsedJson.hasField(json::IS_ADVERT)) {
        isAdvert_ = parsedJson[json::IS_ADVERT].as<bool>()
            ? IsAdvert::Yes
            : IsAdvert::No;
    }
    if (parsedJson.hasField(json::IS_PLATINUM)) {
        isPlatinum_ = parsedJson[json::IS_PLATINUM].as<bool>()
            ? IsPlatinum::Yes
            : IsPlatinum::No;
    }
    if (parsedJson.hasField(json::HAS_OWNER)) {
        hasOwner_ = parsedJson[json::HAS_OWNER].as<bool>()
            ? HasOwner::Yes
            : HasOwner::No;
    }
    if (parsedJson.hasField(json::INDOOR_PLAN_ID)) {
        indoorPlanId_ = boost::lexical_cast<ObjectId>(
            parsedJson[json::INDOOR_PLAN_ID].as<std::string>());
    }
    if (parsedJson.hasField(json::IS_GEOPRODUCT)) {
        isGeoproduct_ = parsedJson[json::IS_GEOPRODUCT].as<bool>()
            ? IsGeoproduct::Yes
            : IsGeoproduct::No;
    }
} catch (const std::exception& ex) {
    ERROR() << "Exception: " << ex.what() << "\n Failed to parse json string: " << jsonString;
    throw;
} catch (...) {
    ERROR() << "Unknown exception while parsing json string: " << jsonString;
    throw;
}


std::string
FeedObjectData::toJson() const
{
    maps::json::Builder builder;
    builder << [&](maps::json::ObjectBuilder objectBuilder) {
        objectBuilder[json::ID] = std::to_string(nmapsId_);
        objectBuilder[json::PERMALINK] = std::to_string(permalink_);
        objectBuilder[json::TO_DELETE] = toDelete_;
        objectBuilder[json::REVISION] =
            std::to_string(chrono::sinceEpoch<std::chrono::milliseconds>(actualizationDate_));

        if (toDelete_) {
            return;
        }
        if (pos_) {
            objectBuilder[json::LON] = pos_->lon;
            objectBuilder[json::LAT] = pos_->lat;
        }
        if (rubricId_) {
            objectBuilder[json::RUBRIC] = *rubricId_;
        }
        objectBuilder[json::SECONDARY_RUBRICS] = [&](maps::json::ArrayBuilder rubricsBuilder) {
            for (const auto& rubricId : secondaryRubricIds_) {
                rubricsBuilder << rubricId;
            };
        };
        if (ftTypeId_) {
            objectBuilder[json::FT_TYPE_ID] = *ftTypeId_;
        }
        objectBuilder[json::NAMES] = [&](maps::json::ArrayBuilder namesBuilder) {
            for (const auto& name : names_) {
                namesBuilder << [&](maps::json::ObjectBuilder nameBuilder) {
                    nameBuilder[json::LANG] = name.lang;
                    nameBuilder[json::NAME] = name.value;
                };
            };
        };
        objectBuilder[json::SHORT_NAMES] = [&](maps::json::ArrayBuilder namesBuilder) {
            for (const auto& name : shortNames_) {
                namesBuilder << [&](maps::json::ObjectBuilder nameBuilder) {
                    nameBuilder[json::LANG] = name.lang;
                    nameBuilder[json::NAME] = name.value;
                };
            };
        };
        if (!indoorLevelUniversal_.empty()) {
            objectBuilder[json::INDOOR_LEVEL_UNIVERSAL] = indoorLevelUniversal_;
        }
        if (isAdvert_) {
            objectBuilder[json::IS_ADVERT] = (isAdvert_ == IsAdvert::Yes);
        }
        if (isPlatinum_) {
            objectBuilder[json::IS_PLATINUM] = (isPlatinum_ == IsPlatinum::Yes);
        }
        if (hasOwner_) {
            objectBuilder[json::HAS_OWNER] = (hasOwner_ == HasOwner::Yes);
        }
        if (indoorPlanId_) {
            objectBuilder[json::INDOOR_PLAN_ID] = std::to_string(*indoorPlanId_);
        }
        if (isGeoproduct_) {
            objectBuilder[json::IS_GEOPRODUCT] = (isGeoproduct_ == IsGeoproduct::Yes);
        }
    };
    return builder.str();
}

double
FeedObjectData::Position::distanceMeters(const FeedObjectData::Position& other) const
{
    const geolib3::Point2 pt1 {this->lon, this->lat};
    const geolib3::Point2 pt2 {other.lon, other.lat};
    return geolib3::geoDistance(pt1, pt2);
}

bool
FeedObjectData::Position::operator !=(const FeedObjectData::Position& other) const
{
    return distanceMeters(other) > POSITION_COMPARE_TOLERANCE_METERS;
}

geolib3::BoundingBox
FeedObjectData::Position::boundingBox() const
{
    return geolib3::BoundingBox(geolib3::geoPoint2Mercator({lon, lat}),
        POSITION_COMPARE_TOLERANCE_METERS, POSITION_COMPARE_TOLERANCE_METERS);
}
} // maps::wiki::poi_feed

