#include <maps/wikimap/mapspro/libs/poi_feed/include/reserved_id.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/common/pg_utils.h>

#include <vector>
using namespace std::string_literals;

namespace maps::wiki::poi_feed {
namespace {
const auto RESERVED_PERMALINKS_MAPPING_TABLE = "sprav.export_poi_reserved_permalinks_mapping"s;
const auto RESERVED_PERMALINKS_UNUSED_TABLE = "sprav.export_poi_reserved_permalinks_unused"s;

std::unordered_map<ObjectId, PermalinkId>
getExistingMapping(
    pqxx::transaction_base& socialWriteableTxn,
    const std::unordered_set<ObjectId>& oids)
{
    std::unordered_map<ObjectId, PermalinkId> objectIdToPermalink;
    objectIdToPermalink.reserve(oids.size());
    auto queryExisting =
        "SELECT object_id, permalink_id FROM " +
        RESERVED_PERMALINKS_MAPPING_TABLE + " WHERE " +
        common::whereClause("object_id", oids);
    auto rows = socialWriteableTxn.exec(queryExisting);
    for (const auto& row : rows) {
        objectIdToPermalink.emplace(
            row["object_id"].as<ObjectId>(),
            row["permalink_id"].as<PermalinkId>());
    }
    return objectIdToPermalink;
}

std::vector<PermalinkId>
reservePermalinkIds(pqxx::transaction_base& socialWriteableTxn, size_t count)
{
    auto queryNewIds =
        "DELETE FROM " +
        RESERVED_PERMALINKS_UNUSED_TABLE +
        " WHERE ctid = ANY(ARRAY(SELECT ctid FROM " +
        RESERVED_PERMALINKS_UNUSED_TABLE +
        " LIMIT " + std::to_string(count) +
        ")) RETURNING permalink_id";
    auto rowsWithNewIds = socialWriteableTxn.exec(queryNewIds);
    REQUIRE(count == rowsWithNewIds.size(),
        "Can't get " << count <<
        " new ids, only " << rowsWithNewIds.size() << " are available.");
    std::vector<PermalinkId> permalinkIds;
    permalinkIds.reserve(count);
    for (const auto& row : rowsWithNewIds) {
        permalinkIds.push_back(row[0].as<PermalinkId>());
    }
    return permalinkIds;
}

std::unordered_map<ObjectId, PermalinkId>
mapMissingPermalinkIds(
    pqxx::transaction_base& socialWriteableTxn,
    const std::unordered_set<ObjectId>& oids,
    const std::unordered_map<ObjectId, PermalinkId>& existingObjectIdToPermalink,
    const std::vector<PermalinkId>& permalinkIds)
{
    const auto count = permalinkIds.size();
    std::vector<ObjectId> notFound;
    notFound.reserve(count);
    for (const auto oid : oids) {
        if (!existingObjectIdToPermalink.count(oid)) {
            notFound.push_back(oid);
        }
    }
    ASSERT(notFound.size() == count);
    std::unordered_map<ObjectId, PermalinkId> newObjectIdToPermalink;
    newObjectIdToPermalink.reserve(count);
    for (size_t i = 0; i < count; ++i) {
        newObjectIdToPermalink.emplace(notFound[i], permalinkIds[i]);
    }
    std::string insertQuery = "INSERT INTO " + RESERVED_PERMALINKS_MAPPING_TABLE +
            " (object_id, permalink_id) VALUES " +
            common::join(
                newObjectIdToPermalink,
                [](const auto& objectIdPermalink) {
                    return
                        "(" +
                        std::to_string(objectIdPermalink.first) +
                        "," +
                        std::to_string(objectIdPermalink.second) +
                        ")";
                }, ',');
    socialWriteableTxn.exec(insertQuery);
    return newObjectIdToPermalink;
}
} // namespace

std::unordered_map<ObjectId, PermalinkId>
assignReservedPermalinkId(
    pqxx::transaction_base& socialWriteableTxn,
    const std::unordered_set<ObjectId>& oids)
{
    if (oids.empty()) {
        return {};
    }
    auto objectIdToPermalink = getExistingMapping(socialWriteableTxn, oids);
    const auto notFoundCount = oids.size() - objectIdToPermalink.size();
    if (!notFoundCount) {
        return objectIdToPermalink;
    }
    auto newPermalinkIds = reservePermalinkIds(socialWriteableTxn, notFoundCount);
    const auto newObjectIdToPermalink = mapMissingPermalinkIds(
        socialWriteableTxn,
        oids,
        objectIdToPermalink,
        newPermalinkIds);
    objectIdToPermalink.insert(newObjectIdToPermalink.begin(), newObjectIdToPermalink.end());
    return objectIdToPermalink;
}

void clearAssignmentsOfReservedPermalinkId(
    pqxx::transaction_base& socialWriteableTxn,
    const std::unordered_set<ObjectId>& oids)
{
    if (oids.empty()) {
        return;
    }
    socialWriteableTxn.exec(
        "DELETE FROM " + RESERVED_PERMALINKS_MAPPING_TABLE +
        " WHERE " + common::whereClause("object_id"s, oids));
}
} // namespace maps::wiki::poi_feed
