#include <maps/wikimap/mapspro/libs/acl/include/aclgateway.h>
#include <maps/wikimap/mapspro/libs/acl/include/exception.h>
#include <maps/wikimap/mapspro/libs/acl/include/subject_path.h>
#include <yandex/maps/wiki/common/pyutils.hpp>

#include <maps/libs/pgpool/include/pgpool3.h>
#include <maps/libs/log8/include/log8.h>

#include <boost/python.hpp>

#include <vector>

namespace bp = boost::python;
namespace mw = maps::wiki;

namespace maps {
namespace wiki {
namespace {

bool isPermissionGrantedInternal(
        pgpool3::Pool& pgPool,
        const std::string& permissionId,
        acl::UID uid,
        const std::vector<std::string>& aoiWkbs)
{
    auto txn = pgPool.masterWriteableTransaction();
    try {
        acl::CheckContext context(uid, aoiWkbs, *txn, {acl::User::Status::Active});
        (acl::SubjectPath(permissionId)).check(context);
    } catch (const acl::ACLException& ex) {
        WARN() << "permission '" << permissionId << "' not granted for uid: " << uid
               << " reason: " << ex.what();
        return false;
    }
    return true;
}

bool isPermissionGranted(
        pgpool3::Pool& pgPool,
        const std::string& permissionId,
        acl::UID uid,
        const bp::object& aoiPyObject)
{
    auto aoiOptional = common::toOptional<std::string>(aoiPyObject);
    std::vector<std::string> aoiWkbs;
    if (aoiOptional) {
        aoiWkbs.push_back(*aoiOptional);
    }

    return isPermissionGrantedInternal(pgPool, permissionId, uid, aoiWkbs);
}

bool isPermissionGrantedMulti(
        pgpool3::Pool& pgPool,
        const std::string& permissionId,
        acl::UID uid,
        const bp::object& aoiPyObject)
{
    return isPermissionGrantedInternal(
        pgPool,
        permissionId,
        uid,
        common::toVector<std::string>(aoiPyObject));
}

std::string loginByUid(
    pgpool3::Pool& pgPool,
    acl::UID uid)
{
    auto txn = pgPool.slaveTransaction();
    return acl::ACLGateway(*txn).user(uid).login();
}

bool isMemberOfGroup(
    pgpool3::Pool& pgPool,
    const std::string& groupName,
    acl::UID uid)
{
    auto txn = pgPool.slaveTransaction();
    acl::ACLGateway gw(*txn);
    for (const auto& group : gw.user(uid).groups()) {
        if (group.name() == groupName) {
            return true;
        }
    }
    return false;
}


} // namespace
} // namespace wiki
} // namespace maps

BOOST_PYTHON_MODULE(acl)
{
    PyEval_InitThreads();

    bp::def("is_permission_granted", ::mw::isPermissionGranted,
            (bp::arg("pgpool"),
             "permission_id",
             "uid",
             bp::arg("wkb")=bp::object()));

    bp::def("is_permission_granted_multi", ::mw::isPermissionGrantedMulti,
            (bp::arg("pgpool"),
             "permission_id",
             "uid",
             "wkbs"));

    bp::def("login_by_uid", ::mw::loginByUid,
        (bp::arg("pgpool"),
         "uid"));

    bp::def("is_member_of_group", ::mw::isMemberOfGroup,
            (bp::arg("pgpool"),
             "group_name",
             "uid"));
}
