class LineLimiter(object):
    """
    Assembles lines from consecutive chunks of bytes.

    Lines are separated by `sep`. Line length is limited by `max_line_length`,
    any excess is skipped.
    Chunks must be fed into `process_chunk` method.
    For each full line `on_line_ready` called.
    """
    def __init__(self, on_line_ready, max_line_length=200, sep="\n"):
        """
        `on_line_ready` is a function of 2 arguments, line itself and a boolean flag indicating
        wether line was truncated to `max_line_length` or not.
        """
        self.max_line = max_line_length
        self.sep = sep
        self.sink = on_line_ready
        self.skip = False
        self._reset()

    def _reset(self):
        self.line_chunks = []
        self.line_left = self.max_line

    def _append(self, chunk):
        self.line_chunks.append(chunk)
        self.line_left -= len(chunk)

    def _commit(self, truncated):
        self.sink("".join(self.line_chunks), truncated)
        self._reset()

    def process_chunk(self, chunk):
        """
        `chunk` is a string of arbitrary size,

        possibly containing none or more of `sep`,
        for example read from program stderr.
        """
        while len(chunk):
            if self.skip:
                sep_pos = chunk.find(self.sep)
                if sep_pos == -1:
                    break
                self.skip = False
                chunk = chunk[sep_pos+1:]
            else:
                sep_pos = chunk.find(self.sep, 0, self.line_left + 1)
                if sep_pos != -1:
                    self._append(chunk[0:sep_pos])
                    self._commit(False)
                    chunk = chunk[sep_pos+1:]
                else:
                    excess_pos = self.line_left
                    if excess_pos >= len(chunk):
                        self._append(chunk)
                        break
                    self._append(chunk[:excess_pos])
                    self._commit(True)
                    self.skip = True
                    chunk = chunk[excess_pos:]
