'''This module defines classes providing simple grinder client.'''

import json
from contextlib import closing
from collections import namedtuple
from functools import wraps
from . import states
from ..utils import require

from six.moves import http_client as httplib


STATUS_NEW = 'New'
STATUS_QUEUED = 'Queued'
STATUS_ASSIGNED = 'Assigned'
STATUS_STARTED = 'Started'
STATUS_POSTPONED = 'Postponed'
STATUS_FINISHED = 'Finished'
STATUS_FAILED = 'Failed'
STATUS_CANCELED = 'Canceled'

_STATUS_MAPPING = {
    STATUS_NEW:       states.PENDING,
    STATUS_QUEUED:    states.PENDING,
    STATUS_ASSIGNED:  states.PENDING,
    STATUS_STARTED:   states.STARTED,
    STATUS_FINISHED:  states.SUCCESS,
    STATUS_POSTPONED: states.PENDING,
    STATUS_FAILED:    states.FAILURE,
    STATUS_CANCELED:  states.REVOKED,
}

GrinderTaskId = namedtuple('GrinderTaskId', ['id'])


class GrinderError(Exception):
    '''Base class for exceptions related to Grinder service.'''
    pass


class TaskNotFound(GrinderError):
    '''Raised if user asks to load information about a task that doesn't
    exist.
    '''
    pass


class CrontabNotFound(GrinderError):
    '''Raised if user asks to load information about a crontab that doesn't
    exist.
    '''
    pass


def get_conn(func):
    @wraps(func)
    def wrap(*args, **kwargs):
        with closing(httplib.HTTPConnection(args[0]._host)) as conn:
            kwargs['conn'] = conn
            return func(*args, **kwargs)
    return wrap


class GrinderGateway(object):
    '''Provides basic methods to interact with Grinder HTTP interface.'''
    def __init__(self, host):
        self._host = host

    @get_conn
    def tasks(self, conn):
        '''Return a list of queued task IDs'''
        conn.request('GET', '/tasks')
        res = conn.getresponse()
        require(res.status == httplib.OK,
                GrinderError('Failed to get tasks: {0} {1}'.
                             format(res.status, res.reason)))
        return json.loads(res.read())

    def submit(self, task_args):
        '''Create a new grinder task using specified dict as parameters.
        Return identifier of the created task.
        '''
        if 'type' not in task_args:
            raise ValueError("Task arguments must contain 'type' field")

        with closing(self._get_http_conn()) as conn:
            body = json.dumps(task_args)
            conn.request('POST', '/tasks', body,
                         headers={'Content-Type': 'application/json'})
            res = conn.getresponse()
            if res.status != httplib.OK:
                raise GrinderError('Failed to post request: {0} {1}'.
                                   format(res.status, res.reason))
            obj = json.loads(res.read())
            return GrinderTaskId(obj['id'])

    def tasklog(self, taskid):
        '''Return execution trace of task with specified identifier.'''
        if not taskid:
            raise ValueError('Task id must not be empty')
        with closing(self._get_http_conn()) as conn:
            conn.request('GET', '/tasklog/{0}'.format(taskid))
            res = conn.getresponse()
            if res.status == httplib.NOT_FOUND:
                # TODO(lifted): It is possible that tasklog is not available
                # just after task submit. We need to fix behaviour of
                # HTTP-interface and then raise an error here.
                return []
            if res.status != httplib.OK:
                raise GrinderError('Failed to connect: {0} {1}'.
                                   format(res.status, res.reason))
            return json.loads(res.read())

    def cancel_task(self, taskid):
        '''Requests task cancel.
        Note: it's up to the worker to actually cancel the task.
        Returns True on successful cancel request, False otherwise'''
        CANCELED = 'canceled'

        if not taskid:
            raise ValueError('Task id must not be empty')
        with closing(self._get_http_conn()) as conn:
            conn.request('PUT', '/tasks/{0}/status'.format(taskid), '"canceled"')
            res = conn.getresponse()
            if res.status == httplib.NOT_FOUND:
                raise GrinderError('Task {0} not found'.format(taskid))
            if res.status != httplib.OK:
                raise GrinderError('Failed to perform request: {0} {1}'.
                                   format(res.status, res.reason))
            json_res = json.loads(res.read())
            if CANCELED not in json_res or not isinstance(json_res[CANCELED], bool):
                raise GrinderError('Unexpected response: {0}'.format(json_res))
            return json_res[CANCELED]

    def task_result(self, taskid):
        '''Return result of task with specified identifier.'''
        tasklog = self.tasklog(taskid)
        return tasklog_to_result(tasklog)

    @get_conn
    def crontabs(self, conn):
        '''Return a list of periodical task IDs'''
        conn.request('GET', '/crontab')
        res = conn.getresponse()
        require(res.status == httplib.OK,
                GrinderError('Failed to get crontabs: {0} {1}'.
                             format(res.status, res.reason)))
        return json.loads(res.read())

    @get_conn
    def crontab(self, cron_task_name, conn):
        '''Get metadata by periodical task name'''
        conn.request('GET', '/crontab/{0}'.format(cron_task_name))
        res = conn.getresponse()
        if res.status == httplib.NOT_FOUND:
            raise CrontabNotFound('Crontab {0} not found'.format(cron_task_name))
        require(res.status == httplib.OK,
                GrinderError('Failed to get crontab: {0} {1}'.
                             format(res.status, res.reason)))
        return json.loads(res.read())

    @get_conn
    def put_crontab(self, cron_task_name, args, cron_expr, conn):
        '''Create or update a periodical task'''
        crontab = {'args': args, 'cron_expr': cron_expr}
        conn.request('PUT',
                     '/crontab/{0}'.format(cron_task_name),
                     body=json.dumps(crontab),
                     headers={'Content-Type': 'application/json'})
        res = conn.getresponse()
        require(res.status == httplib.OK,
                GrinderError('Failed to put crontab: {0} {1}'.
                             format(res.status, res.reason)))
        return crontab

    @get_conn
    def delete_crontab(self, cron_task_name, conn):
        '''Remove a periodical task'''
        conn.request('DELETE', '/crontab/{0}'.format(cron_task_name))
        res = conn.getresponse()
        if res.status == httplib.NOT_FOUND:
            raise CrontabNotFound('Crontab {0} not found'.format(cron_task_name))
        require(res.status == httplib.OK,
                GrinderError('Failed to delete crontab: {0} {1}'.
                             format(res.status, res.reason)))

    def _get_http_conn(self):
        '''Return http connection object.'''
        return httplib.HTTPConnection(self._host)

Result = namedtuple('Result', ['state', 'result'])


def tasklog_to_result(tasklog):
    '''Calculate task result from task execution log.'''
    if len(tasklog) < 1:
        return Result(states.PENDING, None)

    sorted_log = sorted(tasklog, key=lambda e: e['ts'])
    last_entry = sorted_log[-1]
    status, msg = last_entry['status'], last_entry.get('msg')
    if status in _STATUS_MAPPING:
        return Result(_STATUS_MAPPING[status], msg)
    else:
        raise ValueError('Invalid task status: {0}'.format(status))
