import sqlalchemy as sa
from sqlalchemy.ext.declarative import declarative_base

from .. import utils
from .. import config
from . import EM, states, grinder

metadata = sa.MetaData()
Base = declarative_base(metadata=metadata)


class Task(Base):
    __tablename__ = 'task'
    __table_args__ = {'schema': 'service'}

    id = sa.Column(sa.BigInteger, primary_key=True)
    type = sa.Column(sa.String)
    progress = sa.Column(sa.Float, default=0.)
    grinder_task_id = sa.Column(sa.String)
    created = sa.Column(sa.DateTime)
    created_by = sa.Column(sa.BigInteger)
    modified = sa.Column(sa.DateTime)
    modified_by = sa.Column(sa.BigInteger)
    finished = sa.Column(sa.DateTime)
    frozen = sa.Column(sa.Boolean, default=False)
    log = sa.orm.deferred(sa.Column(sa.Text, default=''))
    parent_id = sa.Column(sa.BigInteger)
    yt_operation_id = sa.Column(sa.String)
    status = sa.Column(sa.String)  # explicit status, overrides status from GrinderGateway

    _failure_reason = ''
    __mapper_args__ = {'polymorphic_on': type,
                       'polymorphic_identity': 'task'}

    def on_create(self, uid, request=None):
        self.created_by = self.modified_by = uid
        self.created = self.modified = utils.utcnow()
        if request is not None:
            self.parent_id = request.values.get('parent')

    def on_modify(self, uid):
        self.modified_by = uid
        self.modified = utils.utcnow()

    def on_finish(self):
        self.finished = utils.utcnow()

    @property
    def revocable(self):
        return states.precedence(self.state) > states.precedence(states.REVOKED)

    @property
    def resumable(self):
        return self.state == states.FROZEN

    def revoke(self):
        if self.grinder_task_id:
            gateway = grinder.GrinderGateway(config.get_config().grinder_params.host)
            gateway.cancel_task(self.grinder_task_id)
        if self.yt_operation_id:
            self.status = states.REVOKED

    def resume(self):
        raise NotImplementedError

    @property
    def state(self):
        if '_state' not in dir(self):
            if self.status is not None:
                self._state = self.status
            elif self.grinder_task_id is None:
                self._state = states.FAILURE
                self._failure_reason = 'No task launched'
            else:
                gateway = grinder.GrinderGateway(config.get_config().grinder_params.host)
                result = gateway.task_result(self.grinder_task_id)

                if result.state == states.FAILURE and self.frozen:
                    self._state = states.FROZEN
                else:
                    self._state = result.state
                self._failure_reason = result.result
        return self._state

    def set_state(self, state, **meta):
        self._state = state

    def get_ET_brief(self, *args, **kwargs):
        return self._base_ET(self.context_ET_brief, self.result_ET_brief, args, kwargs)

    def get_ET_full(self, *args, **kwargs):
        return self._base_ET(self.context_ET_full, self.result_ET_full, args, kwargs)

    def context_ET_brief(self, *args, **kwargs):
        return None

    def context_ET_full(self, *args, **kwargs):
        return self.context_ET_brief(*args, **kwargs)

    def result_ET_brief(self, *args, **kwargs):
        return None

    def result_ET_full(self, *args, **kwargs):
        return self.result_ET_brief(*args, **kwargs)

    def _base_ET(self, context_ET_getter, result_ET_getter, args, kwargs):
        props = {
            'id': self.id,
            'type': self.type,
            'status': self.state.lower(),
            'revocable': self.revocable,
            'resumable': self.resumable,
            'created': self.created.isoformat(),
            'uid': self.created_by,
            'progress': self.progress
        }
        if self.parent_id:
            props['parent-id'] = self.parent_id

        ret = EM.task(props)

        context_ET = context_ET_getter(*args, **kwargs)
        if context_ET is not None:
            ret.append(EM.context(context_ET))

        if self.state == states.FAILURE:
            ret.append(EM.result(EM.error_result(self._failure_reason,
                                                 status='INTERNAL_ERROR')))
        else:
            result_ET = result_ET_getter(*args, **kwargs)
            if result_ET is not None:
                ret.append(EM.result(result_ET))
        return ret

    def change_parameters(self, session, request):
        raise NotImplementedError
