# -*- coding: utf-8 -*-
# Alexander Artemenko <art@yandex-team.ru>
# 2009, Yandex
import datetime

from lxml.builder import ElementMaker
import six
if six.PY3:
    long = int

DATE_FORMAT = '%Y-%m-%d'
TIME_FORMAT = '%H:%M:%S'
DATETIME_FORMAT = DATE_FORMAT + 'T' + TIME_FORMAT

# Dictionary with type serializers
TYPE_SERIALIZERS = {
    int: lambda e, v: str(v),
    long: lambda e, v: str(v),
    float: lambda e, v: str(v),
    bool: lambda e, v: str(v).lower(),
    datetime.datetime: lambda e, v: v.strftime(DATETIME_FORMAT),
    datetime.date: lambda e, v: v.strftime(DATE_FORMAT),
    datetime.time: lambda e, v: v.strftime(TIME_FORMAT),
}


class DashElementMaker(ElementMaker):
    """ ElementMaker, заменяющий в тегах и именах атрибутов
        подчеркивание на дефис.

        К примеру::

            >>> ET.tostring(EM.test_element(with_attr = 1))
            '<test-element with-attr="1"/>'
    """
    def __call__(self, tag, *children, **attrib):
        return super(DashElementMaker, self).__call__(
            tag.replace('_', '-'),
            *children,
            **dict((key.replace('_', '-'), value)
                   for key, value in six.iteritems(attrib)))

# Custom Element Maker
EM = DashElementMaker(typemap=TYPE_SERIALIZERS)


def dict2et(element, value, element_maker=EM, **kwargs):
    """ Генератор дерева элементов из dict объектов
        любой глубины вложенности::

            >>> d = {
            ...     'blah': 'minor',
            ...     'another_dict': {
            ...         'one': 1,
            ...         'two': 2,
            ...     },
            ... }
            >>> ET.tostring(dict2et('test_element', d))
            '<test-element><another-dict><two>2</two><one>1</one></another-dict><blah>minor</blah></test-element>'

        Так же, dict может содержать списки элементов::

            >>> d = {
            ...     'list': [
            ...         EM.blah('minor'),
            ...         EM.one(1),
            ...     ]
            ... }
            >>> ET.tostring(dict2et('test_element', d))
            '<test-element><list><blah>minor</blah><one>1</one></list></test-element>'

        Пустые, None значения преобразуются в пустые элементы::

            >>> d = dict(blah = None)
            >>> ET.tostring(dict2et('test_element', d))
            '<test-element><blah/></test-element>'


        Кроме этого, dict2et может принимать именованые аргументы и сериализовать их,
        как атрибуты элемента::

            >>> ET.tostring(dict2et('test', {'blah': 'minor'}, user_id=123))
            '<test user-id="123"><blah>minor</blah></test>'
    """
    if isinstance(value, dict):
        return element_maker(
            element,
            *(dict2et(k, v) for k, v in six.iteritems(value)),
            **kwargs
        )
    elif isinstance(value, list):
        return element_maker(element, *value, **kwargs)
    elif value is None:
        return element_maker(element)

    return element_maker(element, value)
