#include "pyresultsgateway.h"
#include "pyvalidator.h"
#include "queue_iterator.hpp"

#include <yandex/maps/wiki/common/pyutils.hpp>
#include <yandex/maps/wiki/validator/storage/issue_creator.h>
#include <yandex/maps/pylogger/helpers.h>
#include <maps/libs/pgpool/include/pgpool3.h>

#include <boost/lexical_cast.hpp>
#include <boost/python.hpp>
#include <functional>
#include <thread>
#include <memory>

namespace mwv = maps::wiki::validator;
namespace common = maps::wiki::common;

namespace {

mwv::Message::Attributes messageAttributes(const mwv::Message& message)
{ return message.attributes(); }

bp::list messageOids(const mwv::Message& message)
{
    bp::list ret;
    for (mwv::RevisionID revId : message.revisionIds()) {
        ret.append(revId.objectId());
    }
    return ret;
}

std::string messageGeomWkb(const mwv::Message& message)
{ return message.geomWkb(); }

bp::object storedMessageExclusionInfo(const mwv::storage::StoredMessageDatum& datum)
{ return common::toPyObject(datum.exclusionInfo()); }

std::string exclusionInfoCreatedAt(const mwv::storage::ExclusionInfo& exclusion)
{ return exclusion.createdAt; }

std::string storedMessageId(const mwv::storage::StoredMessageDatum& datum)
{ return boost::lexical_cast<std::string>(datum.id()); }

std::string moduleInfoName(const mwv::ModuleInfo& moduleInfo)
{ return moduleInfo.name(); }

bp::list moduleInfoCheckIds(const mwv::ModuleInfo& moduleInfo)
{ return common::toPyList(moduleInfo.checkIds()); }

std::string getOrCreateIssue(
    mwv::storage::IssueCreator& issueCreator,
    bp::object pgPool,
    const std::string& messageIdStr,
    mwv::TId branchId,
    const std::string& login,
    const std::string& pageUrl)
{
    auto txn = bp::extract<maps::pgpool3::Pool&>(pgPool)()
        .masterWriteableTransaction();

    mwv::storage::MessageId messageId;
    std::stringstream ss;
    ss << messageIdStr;
    ss >> messageId;

    auto key = issueCreator.getOrCreateIssue(
        *txn, messageId, branchId, login, pageUrl);
    txn->commit();
    return key;
}

} // namespace

// Note: Validation task has been migrated to grinder,
// so this python module is no longer used
BOOST_PYTHON_MODULE(validator)
{
    PyEval_InitThreads();
    using namespace maps::wiki::validator::python;

    bp::enum_<mwv::Severity>("Severity")
        .value("WARN", mwv::Severity::Warning)
        .value("ERROR", mwv::Severity::Error)
        .value("CRIT", mwv::Severity::Critical)
        .value("FATAL", mwv::Severity::Fatal);

    bp::enum_<mwv::RegionType>("RegionType")
        .value("IMPORTANT", mwv::RegionType::Important)
        .value("UNIMPORTANT", mwv::RegionType::Unimportant);

    bp::class_<mwv::Message::Attributes>(
            "MessageAttributes",
            bp::no_init)
        .def_readonly("severity", &mwv::Message::Attributes::severity)
        .def_readonly("check_id", &mwv::Message::Attributes::checkId)
        .def_readonly("description", &mwv::Message::Attributes::description)
        .def_readonly("region_type", &mwv::Message::Attributes::regionType);

    bp::class_<mwv::Message>
        ("Message", bp::no_init)
        .add_property("attributes", &messageAttributes)
        .add_property("geom_wkb", &messageGeomWkb)
        .add_property("oids", &messageOids);

    bp::class_<mwv::storage::ExclusionInfo>
        ("ExclusionInfo", bp::no_init)
        .add_property("created_at", &exclusionInfoCreatedAt)
        .add_property("created_by", &mwv::storage::ExclusionInfo::createdBy);

    bp::class_<mwv::storage::StoredMessageDatum>
        ("StoredMessageDatum", bp::no_init)
        .add_property("id", &storedMessageId)
        .add_property("is_active", &mwv::storage::StoredMessageDatum::isActive)
        .add_property("exclusion_info", &storedMessageExclusionInfo)
        .add_property("is_viewed", &mwv::storage::StoredMessageDatum::isViewed)
        .def("message", &mwv::storage::StoredMessageDatum::message,
             bp::return_internal_reference<>());

    bp::class_<mwv::ResultPtr>
        ("Result", bp::no_init)
        .def("messages",
            maps::wiki::makeQueueIterator<mwv::ResultPtr, mwv::Result::MessageBuffer>(
                [](mwv::ResultPtr& result) {
                    return result->popMessages();
                }))
        .def("store", &storeResult, (bp::arg("pgpool"), "task_id"));

    bp::class_<mwv::ModuleInfo>
        ("ModuleInfo", bp::no_init)
        .add_property("name", &moduleInfoName)
        .add_property("check_ids", &moduleInfoCheckIds);

    bp::class_<mwv::ValidatorConfig, boost::noncopyable>
        ("ValidatorConfig",
            bp::init<std::string>());

    bp::class_<PyValidator, boost::noncopyable>
        ("Validator",
            bp::init<const mwv::ValidatorConfig&>())
        .def("run", &PyValidator::run,
             (bp::arg("checks"),
              "pgpool",
              "branch_id",
              "commit_id",
              bp::arg("aoi_wkb")=bp::object(),
              bp::arg("aoi_coverage_dir")=bp::object(),
              bp::arg("aoi_buffer")=bp::object()))
        .def("enable_cardinality_check", &PyValidator::enableCardinalityCheck)
        .def("init_modules", &PyValidator::initModules)
        .def("modules", &PyValidator::modules);

    bp::class_<PyResultsGateway, boost::noncopyable>(
            "ResultsGateway",
            bp::init<bp::object, mwv::TTaskId>(
                (bp::arg("pgpool"),
                 "task_id")))
        .add_property("task_id", &PyResultsGateway::taskId)
        .def("message_count", &PyResultsGateway::messageCount,
             (bp::arg("severity") = bp::object(),
              bp::arg("check_id") = bp::object(),
              bp::arg("description") = bp::object(),
              bp::arg("region_type") = bp::object()))
        .def("statistics", &PyResultsGateway::statistics,
             (bp::arg("severity") = bp::object(),
              bp::arg("region_type") = bp::object()))
        .def("messages", &PyResultsGateway::messages,
             (bp::arg("rev_pgpool"),
              bp::arg("rev_token"),
              bp::arg("branch_id"),
              bp::arg("offset"),
              bp::arg("limit"),
              bp::arg("user_id"),
              bp::arg("severity") = bp::object(),
              bp::arg("check_id") = bp::object(),
              bp::arg("description") = bp::object(),
              bp::arg("region_type") = bp::object()))
        .def("message_set_viewed", &PyResultsGateway::messageSetViewed,
             (bp::arg("rev_pgpool"),
              bp::arg("branch_id"),
              bp::arg("user_id"),
              bp::arg("message_id")))
        .def("check_ids_with_fatal_errors", &PyResultsGateway::checkIdsWithFatalErrors);

    bp::class_<mwv::storage::IssueCreator, boost::noncopyable>(
        "IssueCreator",
            bp::init<std::string>())
        .def("get_or_create_issue", getOrCreateIssue, (
            bp::arg("pool"),
            bp::arg("message_id"),
            bp::arg("branch_id"),
            bp::arg("login"),
            bp::arg("page_url")));
}
