#include "pyresultsgateway.h"

#include <yandex/maps/wiki/common/pyutils.hpp>
#include <yandex/maps/wiki/revision/branch_manager.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>
#include <yandex/maps/wiki/validator/storage/message_attributes_filter.h>
#include <yandex/maps/pylogger/helpers.h>

#include <boost/python/extract.hpp>

namespace maps {
namespace wiki {
namespace validator {
namespace python {

void storeResult(
        ResultPtr result,
        pgpool3::Pool& pgPool,
        TTaskId taskId)
{
    maps::pylogger::PyThreadSaver ts;
    storage::storeResult(result, pgPool, taskId);
}

PyResultsGateway::PyResultsGateway(
        bp::object pgPoolPyObject,
        TTaskId taskId)
    : pgPoolPyObject_(pgPoolPyObject)
    , transaction_(
        bp::extract<pgpool3::Pool&>(pgPoolPyObject_)().masterReadOnlyTransaction())
    , gateway_(*transaction_, taskId)
{ }

TTaskId PyResultsGateway::taskId() const
{ return gateway_.taskId(); }

size_t PyResultsGateway::messageCount(
        bp::object severity,
        bp::object checkId,
        bp::object description,
        bp::object regionType) const
{
    storage::MessageAttributesFilter filter;
    filter.severity = common::toOptional<Severity>(severity);
    filter.checkId = common::toOptional<TCheckId>(checkId);
    filter.description = common::toOptional<std::string>(description);
    filter.regionType = common::toOptional<RegionType>(regionType);

    return gateway_.messageCount(filter);
}

bp::list PyResultsGateway::statistics(bp::object severity, bp::object regionType) const
{
    storage::MessageAttributesFilter filter;
    filter.severity = common::toOptional<Severity>(severity);
    filter.regionType = common::toOptional<RegionType>(regionType);

    bp::list result;
    for (const auto& pair : gateway_.statistics(filter)) {
        result.append(bp::make_tuple(pair.first, pair.second));
    }
    return result;
}

bp::list PyResultsGateway::messages(
        pgpool3::Pool& revPgPool,
        const std::string& revToken,
        revision::DBID branchId,
        size_t offset, size_t limit,
        revision::UserID uid,
        bp::object severity,
        bp::object checkId,
        bp::object description,
        bp::object regionType) const
{
    storage::MessageAttributesFilter filter;
    filter.severity = common::toOptional<Severity>(severity);
    filter.checkId = common::toOptional<TCheckId>(checkId);
    filter.description = common::toOptional<std::string>(description);
    filter.regionType = common::toOptional<RegionType>(regionType);

    auto revTxn = revPgPool.slaveTransaction(revToken);
    revision::RevisionsGateway revGateway(
        *revTxn,
        revision::BranchManager(*revTxn).load(branchId));
    auto snapshot = revGateway.snapshot(revGateway.headCommitId());

    return common::toPyList(
        gateway_.messages(filter, snapshot, offset, limit, uid));
}

bp::object
PyResultsGateway::messageSetViewed(
        pgpool3::Pool& revPgPool,
        revision::DBID branchId,
        revision::UserID uid,
        const std::string& messageIdStr) const
{
    auto transaction =
        bp::extract<pgpool3::Pool&>(pgPoolPyObject_)().masterWriteableTransaction();
    storage::ResultsGateway gateway(*transaction, gateway_.taskId());
    auto revTxn = revPgPool.slaveTransaction();
    revision::RevisionsGateway revGateway(
        *revTxn,
        revision::BranchManager(*revTxn).load(branchId));
    auto snapshot = revGateway.snapshot(revGateway.headCommitId());
    auto messageId = boost::lexical_cast<storage::MessageId>(messageIdStr);
    auto datum = std::optional<storage::StoredMessageDatum>(gateway.messageSetViewed(snapshot, uid, messageId));
    transaction->commit();
    return common::toPyObject(datum);
}

bp::list PyResultsGateway::checkIdsWithFatalErrors() const
{
    return common::toPyList(gateway_.checkIdsWithFatalErrors());
}

} // namespace python
} // namespace validator
} // namespace wiki
} // namespace maps
