#include <maps/wikimap/mapspro/libs/query_builder/include/common.h>

#include <maps/wikimap/mapspro/libs/common/include/yandex/maps/wiki/common/string_utils.h>

namespace maps::wiki::query_builder {

constexpr maps::enum_io::Representations<Relation> RELATION_REPRESENTATION {
    {Relation::Ilike,          " ILIKE "},
    {Relation::Less,           " < "},
    {Relation::LessOrEqual,    " <= "},
    {Relation::Like,           " LIKE "},
    {Relation::Greater,        " > "},
    {Relation::GreaterOrEqual, " >= "},
    {Relation::Equal,          " = "},
    {Relation::NotEqual,       " != "},
    {Relation::Intersects,     " && "},
};
DEFINE_ENUM_IO(Relation, RELATION_REPRESENTATION);

std::string QuotedValue::quote(const pqxx::transaction_base& txn) const
{
    switch (needQuote) {
    case Quote::No:
        return value;
    case Quote::Yes:
        return txn.quote(value);
    case Quote::Raw:
        return txn.quote_raw(value);
    }
}

void ModifyData::append(std::string key, std::string value)
{
    appendNoDuplicate(
        std::move(key), {std::move(value), Quote::No});
}

void ModifyData::appendQuoted(std::string key, std::string value)
{
    appendNoDuplicate(
        std::move(key), {std::move(value), Quote::Yes});
}

void ModifyData::appendRawQuoted(std::string key, std::string value)
{
    appendNoDuplicate(
        std::move(key), {std::move(value), Quote::Raw});
}

void ModifyData::appendJsonbEntry(std::string key, JsonbEntry jsonbEntry)
{
    REQUIRE(
        keyValues_.count(key) == 0 &&
        jsonbEntries_.try_emplace(
            std::move(key),
            std::move(jsonbEntry)
        ).second,
        "Cannot append key for which query builder already has value");
}

void ModifyData::appendNoDuplicate(std::string key, QuotedValue quotedValue)
{
    REQUIRE(
        jsonbEntries_.count(key) == 0 &&
        keyValues_.try_emplace(
            std::move(key),
            std::move(quotedValue)
        ).second,
        "Cannot append key for which query builder already has value");
}

void ModifyData::setReturning(std::vector<std::string> columns)
{
    REQUIRE(
        !returningColumns_.size(),
        "Returning columns already set"
    );

    returningColumns_ = std::move(columns);
}

std::string joinJsonbPath(
    const std::string& column,
    const std::vector<std::string>& path)
{
    REQUIRE(path.size() >= 1, "Too short jsonb path");

    std::ostringstream result;
    result << column
        << (path.size() > 1 ? "->" : "")
        << maps::wiki::common::join(
            path.begin(), std::prev(path.end()),
            [](const std::string& token) { return "'" + token + "'"; },
            "->"
        )
        << "->>'" << path.back() << "'";
    return result.str();
}

constexpr maps::enum_io::Representations<JsonbType> JSONB_TYPE_REPRESENTATION {
    {JsonbType::Text, "text"},
    {JsonbType::BigInt,  "bigint"}
};
DEFINE_ENUM_IO(JsonbType, JSONB_TYPE_REPRESENTATION);

std::string makeJsonbSetExpression(
    const pqxx::transaction_base& txn,
    const std::string& column,
    const JsonbEntry& jsonbEntry)
{
    REQUIRE(!jsonbEntry.path.empty(), "Too short jsonb path");

    auto subpath = maps::wiki::common::join(jsonbEntry.path, ",");
    std::ostringstream expression;
    expression << "jsonb_set(" << column << ", '{" << subpath  << "}', "
        << "to_jsonb(" << txn.quote(jsonbEntry.value)
        << "::" << std::string{toString(jsonbEntry.type)} << "))";
    return expression.str();
}

} // namespace maps::wiki::query_builder
