#include <maps/wikimap/mapspro/libs/query_builder/include/where_conditions.h>

#include <maps/wikimap/mapspro/libs/common/include/yandex/maps/wiki/common/string_utils.h>

namespace maps::wiki::query_builder {

constexpr maps::enum_io::Representations<AppendType> APPEND_TYPE_REPRESENTATION {
    {AppendType::And, " AND "},
    {AppendType::Or,  " OR "},
};
DEFINE_ENUM_IO(AppendType, APPEND_TYPE_REPRESENTATION);

namespace {

template<class Conditions, class Lambda>
bool appendConditions(
    std::ostringstream& statement,
    bool isStreamEmpty,
    AppendType appendType,
    const Conditions& conditions,
    const Lambda& conditionToStr)
{
    if (!conditions.empty()) {
        if (!isStreamEmpty) {
            statement << appendType;
        }
        statement << maps::wiki::common::join(
            conditions, conditionToStr, toString(appendType));
        return false;
    }
    return isStreamEmpty;
}

bool appendSubConditions(
    const std::vector<SubConditions>& subConditions,
    std::ostringstream& statement,
    bool isStreamEmpty,
    AppendType appendType,
    const pqxx::transaction_base& txn)
{
    for (const SubConditions& subConditionsItem : subConditions) {
        if (!subConditionsItem.empty()) {
            if (!isStreamEmpty) {
                statement << appendType;
            }
            statement << subConditionsItem.asString(txn);
            isStreamEmpty = false;
        }
    }
    return isStreamEmpty;
}

} // namespace

template <class T>
T& internal::BaseWhereConditions<T>::append(
    std::string key, std::string value, Relation relation)
{
    conditions_.emplace_back(
        std::move(key),
        QuotedValue{std::move(value), Quote::No},
        relation);
    return static_cast<T&>(*this);
}

template <class T>
T& internal::BaseWhereConditions<T>::appendQuoted(
    std::string key, std::string value, Relation relation)
{
    conditions_.emplace_back(
        std::move(key),
        QuotedValue{std::move(value), Quote::Yes},
        relation);
    return static_cast<T&>(*this);
}

template <class T>
T& internal::BaseWhereConditions<T>::and_(SubConditions subConditions)
{
    andSubConditions_.emplace_back(std::move(subConditions));
    return static_cast<T&>(*this);
}

template <class T>
T& internal::BaseWhereConditions<T>::or_(SubConditions subConditions)
{
    orSubConditions_.emplace_back(std::move(subConditions));
    return static_cast<T&>(*this);
}

template <class T>
T& internal::BaseWhereConditions<T>::keyInValues(
    std::string key,
    std::set<std::string> values)
{
    REQUIRE(
        !values.empty(),
        "Malformed WhereConditions::keyInValues(): " << key << " IN ()"
            << ". Not empty set of values expected.");
    inConditions_.emplace_back(
        std::move(key),
        std::move(values),
        /*quoted = */ false);
    return static_cast<T&>(*this);
}

template <class T>
T& internal::BaseWhereConditions<T>::keyInQuotedValues(
    std::string key,
    std::set<std::string> values)
{
    REQUIRE(
        !values.empty(),
        "Malformed WhereConditions::keyInValues(): " << key << " IN ()"
            << ". Not empty set of values expected.");
    inConditions_.emplace_back(
        std::move(key),
        std::move(values),
        /*quoted = */ true);
    return static_cast<T&>(*this);
}

template <class T>
T& internal::BaseWhereConditions<T>::isNull(std::string key)
{
    isNullKeys_.insert(std::move(key));
    return static_cast<T&>(*this);
}

template <class T>
T& internal::BaseWhereConditions<T>::notNull(std::string key)
{
    notNullKeys_.insert(std::move(key));
    return static_cast<T&>(*this);
}

template <class T>
T& internal::BaseWhereConditions<T>::textSearch(
    std::string config, std::string text, std::string word)
{
    textSearchConditions_.emplace_back(
        std::move(config), std::move(text), std::move(word));
    return static_cast<T&>(*this);
}

template <class T>
bool internal::BaseWhereConditions<T>::empty() const
{
    return conditions_.empty()
        && inConditions_.empty()
        && isNullKeys_.empty()
        && notNullKeys_.empty()
        && textSearchConditions_.empty()
        && andSubConditions_.empty()
        && orSubConditions_.empty();
}

template <class T>
internal::BaseWhereConditions<T>::Condition::Condition(
    std::string key, QuotedValue value, Relation relation)
    : key(std::move(key))
    , value(std::move(value))
    , relation(relation)
{}

template <class T>
internal::BaseWhereConditions<T>::InCondition::InCondition(
    std::string key, std::set<std::string> values, bool needQuote)
    : key(std::move(key))
    , values(std::move(values))
    , needQuote(needQuote)
{}

template <class T>
internal::BaseWhereConditions<T>::TextSearchCondition::TextSearchCondition(
    std::string config, std::string text, std::string word)
    : config(std::move(config))
    , text(std::move(text))
    , word(std::move(word))
{}

template <class T>
bool internal::BaseWhereConditions<T>::appendRegularConditions(
    std::ostringstream& statement,
    bool isStreamEmpty,
    AppendType appendType,
    const pqxx::transaction_base& txn) const
{
    auto conditionToStr = [&txn](const BaseWhereConditions::Condition& condition) {
        return condition.key + std::string{toString(condition.relation)}
            + condition.value.quote(txn);
    };
    return appendConditions(statement, isStreamEmpty, appendType, conditions_, conditionToStr);
}

template <class T>
bool internal::BaseWhereConditions<T>::appendInConditions(
    std::ostringstream& statement,
    bool isStreamEmpty,
    AppendType appendType,
    const pqxx::transaction_base& txn) const
{
    auto conditionToStr = [&txn](const BaseWhereConditions::InCondition& inCondition) {
        const auto joinedValues = maps::wiki::common::join(
            inCondition.values,
            [&txn, &inCondition](const std::string value) {
                return inCondition.needQuote ? txn.quote(value) : value;
            },
            ", "
        );
        return inCondition.key + " IN (" + joinedValues + ')';
    };
    return appendConditions(statement, isStreamEmpty, appendType, inConditions_, conditionToStr);
}

template <class T>
bool internal::BaseWhereConditions<T>::appendIsNullKeys(
    std::ostringstream& statement,
    bool isStreamEmpty,
    AppendType appendType) const
{
    auto conditionToStr = [](const std::string& key) {
        return key + " IS NULL";
    };
    return appendConditions(statement, isStreamEmpty, appendType, isNullKeys_, conditionToStr);
}

template <class T>
bool internal::BaseWhereConditions<T>::appendNotNullKeys(
    std::ostringstream& statement,
    bool isStreamEmpty,
    AppendType appendType) const
{
    auto conditionToStr = [](const std::string& key) {
        return key + " IS NOT NULL";
    };
    return appendConditions(statement, isStreamEmpty, appendType, notNullKeys_, conditionToStr);
}

template <class T>
bool internal::BaseWhereConditions<T>::appendTextSearchConditions(
    std::ostringstream& statement,
    bool isStreamEmpty,
    AppendType appendType,
    const pqxx::transaction_base& txn) const
{
    auto conditionToStr = [&txn](const BaseWhereConditions::TextSearchCondition& searchCondition) {
        return "to_tsvector(" + txn.quote(searchCondition.config) + ", " +
            searchCondition.text + ") @@ plainto_tsquery(" +
            txn.quote(searchCondition.word) + ")";
    };
    return appendConditions(statement, isStreamEmpty, appendType, textSearchConditions_, conditionToStr);
}

template <class T>
std::string internal::BaseWhereConditions<T>::createStatements(
    AppendType appendType,
    const pqxx::transaction_base& txn) const
{
    bool isStreamEmpty = true;
    std::ostringstream statement;
    isStreamEmpty = appendRegularConditions(statement, isStreamEmpty, appendType, txn);
    isStreamEmpty = appendInConditions(statement, isStreamEmpty, appendType, txn);
    isStreamEmpty = appendIsNullKeys(statement, isStreamEmpty, appendType);
    isStreamEmpty = appendNotNullKeys(statement, isStreamEmpty, appendType);
    isStreamEmpty = appendTextSearchConditions(statement, isStreamEmpty, appendType, txn);
    isStreamEmpty = appendSubConditions(
        andSubConditions_,
        statement,
        isStreamEmpty,
        AppendType::And,
        txn
    );
    isStreamEmpty = appendSubConditions(
        orSubConditions_,
        statement,
        isStreamEmpty,
        AppendType::Or,
        txn
    );
    return statement.str();
}

std::string WhereConditions::asString(const pqxx::transaction_base& txn) const
{
    if (empty()) {
        return {};
    }
    return "WHERE " + createStatements(AppendType::And, txn);
}

SubConditions::SubConditions(AppendType appendType)
    : appendType_{appendType}
{}

std::string SubConditions::asString(const pqxx::transaction_base& txn) const
{
    if (empty()) {
        return "TRUE";
    }
    return "( " + createStatements(appendType_, txn) + " ) ";
}

template class internal::BaseWhereConditions<WhereConditions>;
template class internal::BaseWhereConditions<SubConditions>;

} // namespace maps::wiki::query_builder
