#pragma once

#include <maps/wikimap/mapspro/libs/query_builder/include/common.h>
#include <maps/libs/enum_io/include/enum_io.h>

#include <pqxx/pqxx>
#include <set>
#include <vector>

namespace maps::wiki::query_builder {

enum AppendType {
    And,
    Or
};
DECLARE_ENUM_IO(AppendType);

class SubConditions;

namespace internal {

template <class T>
class BaseWhereConditions {
public:
    BaseWhereConditions() = default;
    BaseWhereConditions(BaseWhereConditions&&) = default;
    BaseWhereConditions(const BaseWhereConditions&) = default;
    BaseWhereConditions& operator=(BaseWhereConditions&&) = default;
    virtual ~BaseWhereConditions() = default;

    T& append(
        std::string key,
        std::string value,
        Relation relation = Relation::Equal);
    T& appendQuoted(
        std::string key,
        std::string value,
        Relation relation = Relation::Equal);
    T& and_(SubConditions);
    T& or_(SubConditions);

    T& keyInValues(
        std::string key,
        std::set<std::string> values);
    T& keyInQuotedValues(
        std::string key,
        std::set<std::string> values);

    T& isNull(std::string key);
    T& notNull(std::string key);
    T& textSearch(
        std::string config,
        std::string text,
        std::string word);

    virtual std::string asString(const pqxx::transaction_base& txn) const = 0;
    bool empty() const;

protected:
    std::string createStatements(
        AppendType appendType,
        const pqxx::transaction_base& txn) const;

private:
    struct Condition {
        std::string key;
        QuotedValue value;
        Relation relation;

        Condition(
            std::string key,
            QuotedValue value,
            Relation realation);
    };

    struct InCondition {
        std::string key;
        std::set<std::string> values;
        bool needQuote;

        InCondition(
            std::string key,
            std::set<std::string> values,
            bool needQuote);
    };

    struct TextSearchCondition {
        std::string config;
        std::string text;
        std::string word;

        TextSearchCondition(
            std::string config,
            std::string text,
            std::string word);
    };

    bool appendRegularConditions(
        std::ostringstream& statement,
        bool isStreamEmpty,
        AppendType appendType,
        const pqxx::transaction_base& txn) const;
    bool appendInConditions(
        std::ostringstream& statement,
        bool isStreamEmpty,
        AppendType appendType,
        const pqxx::transaction_base& txn) const;
    bool appendIsNullKeys(
        std::ostringstream& statement,
        bool isStreamEmpty,
        AppendType appendType) const;
    bool appendNotNullKeys(
        std::ostringstream& statement,
        bool isStreamEmpty,
        AppendType appendType) const;
    bool appendTextSearchConditions(
        std::ostringstream& statement,
        bool isStreamEmpty,
        AppendType appendType,
        const pqxx::transaction_base& txn) const;

    std::vector<Condition> conditions_;
    std::vector<InCondition> inConditions_;

    std::set<std::string> isNullKeys_;
    std::set<std::string> notNullKeys_;
    std::vector<TextSearchCondition> textSearchConditions_;

    std::vector<SubConditions> andSubConditions_;
    std::vector<SubConditions> orSubConditions_;
};

} // namespace internal

class WhereConditions : public internal::BaseWhereConditions<WhereConditions> {
public:
    WhereConditions() = default;
    WhereConditions(WhereConditions&&) = default;
    WhereConditions(const WhereConditions&) = default;
    WhereConditions& operator=(WhereConditions&&) = default;

    std::string asString(const pqxx::transaction_base& txn) const override;
};

class SubConditions : public internal::BaseWhereConditions<SubConditions> {
public:
    explicit SubConditions(AppendType appendType = AppendType::And);
    SubConditions(SubConditions&&) = default;
    SubConditions(const SubConditions&) = default;
    SubConditions& operator=(SubConditions&&) = default;

    std::string asString(const pqxx::transaction_base& txn) const override;

private:
    AppendType appendType_;
};

} // namespace maps::wiki::query_builder
