#include "commit_approver.h"
#include "commit_relations.h"
#include "sql_strings.h"
#include "helpers.h"
#include "query_generator.h"

#include <maps/libs/common/include/exception.h>
#include <sstream>

namespace maps::wiki::revision {

using namespace helpers;

namespace {

const std::string SELECT_LOCK_COMMITS_PREFIX =
    "SELECT " + sql::col::ID + ", " + sql::col::STATE +
    "   FROM " + sql::table::COMMIT +
    "   WHERE ";

const std::string IS_DRAFT =
    sql::col::STATE + " = '" + sql::val::COMMIT_STATE_DRAFT + "'";

const std::string SELECT_LOCK_ALL_COMMITS_PREFIX =
    "SELECT " + sql::col::ID +
    "   FROM " + sql::table::COMMIT +
    "   WHERE " + IS_DRAFT +
    "   AND " + sql::col::ID + " <= ";

const std::string SELECT_LOCK_COMMITS_POSTFIX =
    "   ORDER BY " + sql::col::ID + " FOR UPDATE";


const std::string UPDATE_LOCKED_COMMITS_PREFIX =
    "WITH w0 AS ("
        "SELECT " + sql::func::NEXT_VALUE + "('" + sql::seq::APPROVE_ORDER + "'::regclass)" +
    ")" +
    "UPDATE " + sql::table::COMMIT +
    " SET " +
        sql::col::STATE + " = '" + sql::val::COMMIT_STATE_APPROVED + "', " +
        sql::col::APPROVE_ORDER + " = w0." + sql::col::NEXT_VALUE +
    " FROM w0 " +
    " WHERE " + IS_DRAFT + " AND ";

} // namespace


CommitApprover::CommitApprover(Transaction& work) : work_(work) {}


DBIDSet
CommitApprover::process(const DBIDSet& commitIds) const
{
    if (commitIds.empty()) {
        return {};
    }

    TransactionGuard guard(work_);
    const auto contributingCommitIds = findAllDraftContributingCommits(work_, commitIds, {});
    auto approvingCommitIds = lockCommits(contributingCommitIds);
    processDraftCommits(approvingCommitIds);
    guard.ok();

    return approvingCommitIds;
}

DBIDSet
CommitApprover::processAll(DBID maxCommitId) const
{
    if (!maxCommitId) {
        return {};
    }

    TransactionGuard guard(work_);
    auto approvingCommitIds = lockAllCommits(maxCommitId);
    processDraftCommits(approvingCommitIds);
    guard.ok();

    return approvingCommitIds;
}

void
CommitApprover::processDraftCommits(const DBIDSet& commitIds) const
{
    if (commitIds.empty()) {
        return;
    }

    // writing
    auto queryIds = QueryGenerator::buildFilterByAttrValues(sql::col::ID, commitIds);
    auto r = work_.exec(UPDATE_LOCKED_COMMITS_PREFIX + queryIds); // draft -> approved
    auto count = r.affected_rows();
    REQUIRE(count == commitIds.size(), "failed approving commits: " << queryIds);
}

DBIDSet
CommitApprover::lockAllCommits(DBID maxCommitId) const
{
    std::ostringstream query;
    query << SELECT_LOCK_ALL_COMMITS_PREFIX << maxCommitId <<
             SELECT_LOCK_COMMITS_POSTFIX;

    DBIDSet ids;
    for (const auto& row : work_.exec(query.str())) {
        DBID id = row[sql::col::ID].as<DBID>();
        ASSERT(ids.insert(id).second);
    }
    return ids;
}

DBIDSet
CommitApprover::lockCommits(const DBIDSet& commitIds) const
{
    if (commitIds.empty()) {
        return {};
    }

    auto queryIds = QueryGenerator::buildFilterByAttrValues(sql::col::ID, commitIds);
    const auto rows = work_.exec(
        SELECT_LOCK_COMMITS_PREFIX + queryIds + SELECT_LOCK_COMMITS_POSTFIX);

    DBIDSet draftCommitIds;
    for (const auto& row: rows) {
        DBID commitId = row[sql::col::ID].as<DBID>();
        CommitState state = stringToCommitState(row[sql::col::STATE].as<std::string>());
        if (state == CommitState::Draft) {
            draftCommitIds.insert(commitId);
        }
    }
    return draftCommitIds;
}

} // namespace maps::wiki::revision
