#include <yandex/maps/wiki/revision/common.h>
#include "sql_strings.h"
#include <maps/libs/common/include/exception.h>

#include <strings.h>
#include <ctype.h>

#include <utility>

namespace maps::wiki::revision {

RelationData::RelationData(DBID masterObjectId, DBID slaveObjectId)
    : masterObjectId_(masterObjectId)
    , slaveObjectId_(slaveObjectId)
{
    REQUIRE(masterObjectId_, "invalid masterObjectId");
}

namespace {

template <typename T>
class StringTypeConvertor {
public:
    using InitDataType = std::map<std::string, T>;

    StringTypeConvertor(std::string name, const InitDataType& data)
        : name_(std::move(name))
    {
        for (const auto& pair : data) {
            ASSERT(data_.insert(pair).second);
        }
    }

    T fromString(const std::string& str) const
    {
        if (!str.empty()) {
            auto it = data_.find(str);
            if (it != data_.end()) {
                return it->second;
            }
        }
        throw maps::RuntimeError()
            << "couldn't parse " << name_ << " from string '" << str << "'";
    }

private:
    struct CaseInsensitiveCompare {
        bool operator () (const std::string& left, const std::string& right) const
        {
            return ::strcasecmp(left.c_str(), right.c_str()) < 0;
        }
    };

    std::map<std::string, T, CaseInsensitiveCompare> data_;
    const std::string name_;
};

const StringTypeConvertor<BranchType> BRANCH_TYPE_CONVERTOR(
    "branch type",
    {
        {sql::val::BRANCH_TYPE_TRUNK,    BranchType::Trunk},
        {sql::val::BRANCH_TYPE_APPROVED, BranchType::Approved},
        {sql::val::BRANCH_TYPE_STABLE,   BranchType::Stable},
        {sql::val::BRANCH_TYPE_ARCHIVE,  BranchType::Archive},
        {sql::val::BRANCH_TYPE_DELETED,  BranchType::Deleted}
    });

const StringTypeConvertor<BranchState> BRANCH_STATE_CONVERTOR(
    "branch state",
    {
        {sql::val::BRANCH_STATE_UNAVAILABLE, BranchState::Unavailable},
        {sql::val::BRANCH_STATE_NORMAL,      BranchState::Normal},
        {sql::val::BRANCH_STATE_PROGRESS,    BranchState::Progress}
    });


} // namespace


std::ostream& operator << (std::ostream& os, CommitState commitState)
{
    switch (commitState) {
    case CommitState::Draft:
        os << sql::val::COMMIT_STATE_DRAFT;
        break;
    case CommitState::Approved:
        os << sql::val::COMMIT_STATE_APPROVED;
        break;
    }
    return os;
}


std::istream& operator >> (std::istream& is, BranchType& type)
{
    std::string str;
    is >> str;
    type = BRANCH_TYPE_CONVERTOR.fromString(str);
    return is;
}


std::ostream& operator << (std::ostream& os, BranchType type)
{
    switch (type) {
        case BranchType::Trunk:    os << sql::val::BRANCH_TYPE_TRUNK;    break;
        case BranchType::Approved: os << sql::val::BRANCH_TYPE_APPROVED; break;
        case BranchType::Stable:   os << sql::val::BRANCH_TYPE_STABLE;   break;
        case BranchType::Archive:  os << sql::val::BRANCH_TYPE_ARCHIVE;  break;
        case BranchType::Deleted:  os << sql::val::BRANCH_TYPE_DELETED;  break;
    }
    return os;
}

std::istream& operator >> (std::istream& is, BranchState& state)
{
    std::string str;
    is >> str;
    state = BRANCH_STATE_CONVERTOR.fromString(str);
    return is;
}

std::ostream& operator << (std::ostream& os, BranchState state)
{
    switch (state) {
        case BranchState::Unavailable: os << sql::val::BRANCH_STATE_UNAVAILABLE; break;
        case BranchState::Normal:      os << sql::val::BRANCH_STATE_NORMAL;      break;
        case BranchState::Progress:    os << sql::val::BRANCH_STATE_PROGRESS;    break;
    }
    return os;
}

} // namespace maps::wiki::revision
