#include <yandex/maps/wiki/revision/diff.h>
#include "helpers.h"
#include "result_interpreter.h"
#include "sql_strings.h"
#include <boost/format.hpp>
#include <maps/libs/common/include/exception.h>
#include <memory>
#include <utility>
#include <yandex/maps/wiki/revision/objectrevision.h>

namespace maps::wiki::revision {

namespace {
const Attributes EMPTY_ATTRIBUTES;
const std::string EMPTY_STRING;
const Description EMPTY_DESCRIPTION;
const Wkb EMPTY_GEOMETRY;

const auto QUERY_ENTIRE_COMMIT = boost::format(
    "WITH r1 AS (SELECT * FROM revision.object_revision WHERE commit_id = %1%)"
    " SELECT * FROM r1"
    " UNION"
    " SELECT old.* FROM revision.object_revision old, r1"
    " WHERE old.commit_id = r1.prev_commit_id"
    " AND old.object_id = r1.object_id"
    );

const auto QUERY_COMMIT_OBJECT = boost::format(
    "WITH r1 AS (SELECT * FROM revision.object_revision WHERE commit_id = %1% AND "
    " (object_id = %2% OR slave_object_id = %2% OR master_object_id = %2%))"
    " SELECT * FROM r1"
    " UNION"
    " SELECT old.* FROM revision.object_revision old, r1"
    " WHERE old.commit_id = r1.prev_commit_id"
    " AND old.object_id = r1.object_id"
    " AND (old.object_id = %2% OR old.slave_object_id = %2% OR old.master_object_id = %2%)"
    );

template <typename T>
std::optional<Diff<T>>
createDiff(const T& oldValue, const T& newValue)
{
    if (oldValue != newValue) {
        return Diff<T> {oldValue, newValue};
    }
    return std::optional< Diff<T> >();
}

void
initData(
    ObjectRevisionDiff::Data& data,
    const ObjectData& newData, const ObjectData* oldData)
{
    if (oldData && oldData->deleted != newData.deleted) {
        data.deleted = DeletedDiff(oldData->deleted, newData.deleted);
    }

    if (!oldData) {
        data.newRelationData = newData.relationData;
    }
    else if (!newData.relationData || !oldData->relationData ||
        *newData.relationData != *oldData->relationData) {

        data.newRelationData = newData.relationData;
        data.oldRelationData = oldData->relationData;
    }

    data.description = createDiff(
        oldData && oldData->description ? *oldData->description : EMPTY_DESCRIPTION,
        newData.description ? *newData.description : EMPTY_DESCRIPTION);

    data.geometry = createDiff(
        oldData && oldData->geometry ? *oldData->geometry: EMPTY_GEOMETRY,
        newData.geometry ? *newData.geometry : EMPTY_GEOMETRY);

    data.attributes = createAttributesDiff(
        oldData && oldData->attributes ? *oldData->attributes : EMPTY_ATTRIBUTES,
        newData.attributes ? *newData.attributes : EMPTY_ATTRIBUTES);
}
} // namespace

class ObjectRevisionDiffImpl {
public:
    explicit ObjectRevisionDiffImpl(ObjectRevisionDiff::Data  data, const std::optional<RevisionID>& oldId, const RevisionID& newId);
    explicit ObjectRevisionDiffImpl(const RevisionID& unionRevId);
    explicit ObjectRevisionDiffImpl(const ObjectRevision& newRev);
    ObjectRevisionDiffImpl(const ObjectRevision& newRev, const ObjectRevision& oldRev);

    const ObjectRevisionDiff::Data& data() const { return data_; }

    const RevisionID& newId() const { return newId_; }
    const RevisionID& oldId() const { return oldId_; }

private:
    ObjectRevisionDiff::Data data_;
    RevisionID oldId_; // empty - for new data, first data of pairs should be skipped
    RevisionID newId_;
};



ObjectRevisionDiff::ObjectRevisionDiff(
        const Data& data,
        const std::optional<RevisionID>& oldId,
        const RevisionID& newId)
    : impl_(new ObjectRevisionDiffImpl(data, oldId, newId))
{
}

ObjectRevisionDiffImpl::ObjectRevisionDiffImpl(
        ObjectRevisionDiff::Data data,
        const std::optional<RevisionID>& oldId,
        const RevisionID& newId)
    : data_(std::move(data))
    , newId_(newId)
{
    if (oldId) {
        oldId_ = *oldId;
    }
}

ObjectRevisionDiff::ObjectRevisionDiff(
    const ObjectRevision& newRev)
    : impl_(new ObjectRevisionDiffImpl(newRev))
{
}

ObjectRevisionDiffImpl::ObjectRevisionDiffImpl(
    const ObjectRevision& newRev)
    : newId_(newRev.id())
{
    initData(data_, newRev.data(), 0);
}

ObjectRevisionDiff::ObjectRevisionDiff(
    const ObjectRevision& newRev, const ObjectRevision& oldRev)
    : impl_(new ObjectRevisionDiffImpl(newRev, oldRev))
{
}

ObjectRevisionDiffImpl::ObjectRevisionDiffImpl(
    const ObjectRevision& newRev, const ObjectRevision& oldRev)
    : oldId_(oldRev.id())
    , newId_(newRev.id())
{
    initData(data_, newRev.data(), &(oldRev.data()));
}

ObjectRevisionDiff::ObjectRevisionDiff(
   const RevisionID& unionRevId)
   : impl_(new ObjectRevisionDiffImpl(unionRevId))
{
}

ObjectRevisionDiffImpl::ObjectRevisionDiffImpl(
   const RevisionID& unionRevId)
   : oldId_(unionRevId)
   , newId_(unionRevId)
{
   REQUIRE(unionRevId.empty() || unionRevId.valid(), "revision id must be empty or valid");
}

ObjectRevisionDiff::ObjectRevisionDiff(const ObjectRevisionDiff& other)
    : impl_(new ObjectRevisionDiffImpl(*other.impl_))
{
}

ObjectRevisionDiff::ObjectRevisionDiff(ObjectRevisionDiff&&) noexcept = default;

ObjectRevisionDiff&
ObjectRevisionDiff::operator =(const ObjectRevisionDiff& other)
{
    if (&other != this) {
        impl_ = std::make_unique<ObjectRevisionDiffImpl>(*other.impl_);
    }
    return *this;
}

ObjectRevisionDiff&
ObjectRevisionDiff::operator =(ObjectRevisionDiff&&) noexcept = default;

ObjectRevisionDiff::~ObjectRevisionDiff() = default;

const ObjectRevisionDiff::Data&
ObjectRevisionDiff::data() const
{
    return impl_->data();
}

const RevisionID&
ObjectRevisionDiff::newId() const
{
    return impl_->newId();
}

const RevisionID&
ObjectRevisionDiff::oldId() const
{
    return impl_->oldId();
}

namespace {
struct RevisionDataBeforeAndAfter
{
    RevisionDataBeforeAndAfter() = default;
    DBID prevCommitId{0};
    std::optional<RevisionData> before;
    std::optional<RevisionData> after;
};

using RevisionDataBeforeAndAfterById = std::map<DBID, RevisionDataBeforeAndAfter>;
using OnUpdateFunc = std::function<void (const RevisionData &, const RevisionData &, DBID, DBID)>;
using OnExistanceChange = std::function<void (const RevisionData &, DBID, DBID)>;

void
forAllRecords(
    const RevisionDataBeforeAndAfterById& revisionFields,
    const OnUpdateFunc& onUpdate,
    const OnExistanceChange& onCreate,
    const OnExistanceChange& onDelete)
{
    for (const auto& rfPc : revisionFields) {
        const auto& before = rfPc.second.before;
        const auto& after = *rfPc.second.after;
        if (before && !before->deleted && !after.deleted) {
            onUpdate(*before, after, rfPc.first, rfPc.second.prevCommitId);
        } else if (before && !before->deleted) {
            onDelete(*before, rfPc.first, rfPc.second.prevCommitId);
        } else if (!after.deleted) {
            onCreate(after, rfPc.first, rfPc.second.prevCommitId);
        }
    }
}

class Resources
{
public:
    Resources(const RevisionDataBeforeAndAfterById& revs, pqxx::transaction_base& txn)
    {
        DBIDSet geomIds;
        DBIDSet attrIds;
        DBIDSet descrIds;
        auto onUpdateRes = [&](const RevisionData& before, const RevisionData& after, DBID /*objectId*/, DBID /*prevCommitId*/) {
                if (before.attributesId != after.attributesId) {
                    attrIds.insert(before.attributesId);
                    attrIds.insert(after.attributesId);
                }
                if (before.geometryId != after.geometryId) {
                    geomIds.insert(before.geometryId);
                    geomIds.insert(after.geometryId);
                }
                if (before.descriptionId != after.descriptionId) {
                    descrIds.insert(before.descriptionId);
                    descrIds.insert(after.descriptionId);
                }
            };
        auto onExistanceChangeRes = [&](const RevisionData& fields, DBID /*objectId*/, DBID /*prevCommitId*/) {
                if (fields.attributesId) {
                    attrIds.insert(fields.attributesId);
                }
                if (fields.geometryId) {
                    geomIds.insert(fields.geometryId);
                }
                if (fields.descriptionId) {
                    descrIds.insert(fields.descriptionId);
                }
            };
        forAllRecords(revs, onUpdateRes, onExistanceChangeRes, onExistanceChangeRes);
        loadGeometries(geomIds, txn);
        loadAttributes(attrIds, txn);
        loadDescriptions(descrIds, txn);
    }

    template <typename T> const T& getResource(DBID resId, const std::map<DBID, T>& resourceStore) const
    {
        ASSERT(resId);
        auto it = resourceStore.find(resId);
        ASSERT(it != resourceStore.end());
        return it->second;
    }
    const Wkb& geometry(DBID id) const
    {
        return getResource<Wkb>(id, wkbById_);
    }
    const Attributes& attributes(DBID id) const
    {
        return getResource<Attributes>(id, attrsById_);
    }
    const Description& description(DBID id) const
    {
        return getResource<Description>(id, descrById_);
    }

private:
    void
    loadGeometries(const DBIDSet& geomIds, pqxx::transaction_base& txn)
    {
        if (geomIds.empty()) {
            return;
        }
        auto wkbs = txn.exec(
            "SELECT id, " + sql::func::GEOM_TO_BINARY +
            "(contents) as contents FROM revision.geometry WHERE id IN (" +
            helpers::valuesToString(geomIds) + ")");
        for (const auto& row : wkbs) {
            wkbById_.insert({row[sql::col::ID].as<DBID>(), pqxx::binarystring(row[sql::col::CONTENTS]).str()});
        }
    }

    void
    loadAttributes(const DBIDSet& attrIds, pqxx::transaction_base& txn)
    {
        if (attrIds.empty()) {
            return;
        }
        auto attrs = txn.exec(
            "SELECT id, " + sql::func::HSTORE_TO_ARRAY +
            "(contents) as contents FROM revision.attributes WHERE id IN (" +
            helpers::valuesToString(attrIds) + ")");
        for (const auto& row : attrs) {
            attrsById_.insert({row[sql::col::ID].as<DBID>(),
                helpers::stringToAttributes(row[sql::col::CONTENTS].c_str())});
        }
    }

    void
    loadDescriptions(const DBIDSet& descrIds, pqxx::transaction_base& txn)
    {
        if (descrIds.empty()) {
            return;
        }
        auto attrs = txn.exec(
            "SELECT id, contents FROM revision.description WHERE id IN (" +
            helpers::valuesToString(descrIds) + ")");
        for (const auto& row : attrs) {
            descrById_.insert({row[sql::col::ID].as<DBID>(), row[sql::col::CONTENTS].c_str()});
        }
    }

    std::map<DBID, Wkb> wkbById_;
    std::map<DBID, Attributes> attrsById_;
    std::map<DBID, Description> descrById_;
};

RevisionDataBeforeAndAfterById
loadDiffRows(DBID commitId, DBID objectId, pqxx::transaction_base& txn)
{
    const auto& revsQuery = str(objectId
        ? (boost::format(QUERY_COMMIT_OBJECT) % commitId % objectId)
        : (boost::format(QUERY_ENTIRE_COMMIT) % commitId));
    const auto& revRecords = txn.exec(revsQuery);
    RevisionDataBeforeAndAfterById oidToRevDataDiff;
    for (const auto& row : revRecords) {
        RevisionData revFlds = loadRevisionDataFromPqxx(row);
        DBID rowObjectId = row[sql::col::OBJECT_ID].as<DBID>();
        DBID rowCommitId = row[sql::col::COMMIT_ID].as<DBID>();
        if (rowCommitId == commitId) {
            oidToRevDataDiff[rowObjectId].after = revFlds;
        } else if (!revFlds.deleted) {
            oidToRevDataDiff[rowObjectId].before = revFlds;
            oidToRevDataDiff[rowObjectId].prevCommitId = rowCommitId;
        }
    }
    return oidToRevDataDiff;
}

}//namespace


CommitDiff
commitDiff(pqxx::transaction_base& txn, DBID commitId, DBID objectId /*=0*/)
{
    auto oidToRevDataDiff = loadDiffRows(commitId, objectId, txn);
    Resources resources(oidToRevDataDiff, txn);
    CommitDiff result;
    auto onUpdate = [&](const RevisionData& before, const RevisionData& after, DBID objectId, DBID prevCommitId) {
        ObjectRevisionDiff::Data data;
        if (before.masterObjectId != after.masterObjectId
            || before.slaveObjectId != after.slaveObjectId) {
            data.newRelationData = RelationData(after.masterObjectId, after.slaveObjectId);
            data.oldRelationData = RelationData(before.masterObjectId, before.slaveObjectId);
        }
        if (after.geometryId != before.geometryId) {
            data.geometry = GeometryDiff {
                resources.geometry(before.geometryId),
                resources.geometry(after.geometryId)
            };
        }
        if (after.descriptionId != before.descriptionId) {
            data.description = DescriptionDiff {
                resources.description(before.descriptionId),
                resources.description(after.descriptionId)
            };
        }
        if (after.attributesId != before.attributesId) {
            data.attributes = createAttributesDiff(
                resources.attributes(before.attributesId),
                resources.attributes(after.attributesId)
            );
        }
        result.emplace(objectId,
            ObjectRevisionDiff(data,
                RevisionID(objectId, prevCommitId),
                RevisionID(objectId, commitId)));
    };

    auto onCreate = [&](const RevisionData& after, DBID objectId, DBID /*prevCommitId*/) {
        ObjectRevisionDiff::Data data;
        if (after.masterObjectId) {
            data.newRelationData = RelationData(after.masterObjectId, after.slaveObjectId);
        }
        if (after.geometryId) {
            data.geometry = GeometryDiff {
                Wkb(),
                resources.geometry(after.geometryId)
            };
        }
        if (after.attributesId) {
            data.attributes = createAttributesDiff(
                Attributes(),
                resources.attributes(after.attributesId)
            );
        }
        if (after.descriptionId) {
            data.description = DescriptionDiff {
                Description(),
                resources.description(after.descriptionId)
            };
        }
        result.emplace(objectId,
            ObjectRevisionDiff(data,
                std::optional<RevisionID>(),
                RevisionID(objectId, commitId)));
    };

    auto onDelete = [&](const RevisionData& before, DBID objectId, DBID prevCommitId) {
        ObjectRevisionDiff::Data data;
        if (before.masterObjectId) {
            data.oldRelationData = RelationData(before.masterObjectId, before.slaveObjectId);
        }
        data.deleted = DeletedDiff(false, true);
        if (before.geometryId) {
            data.geometry = GeometryDiff {
                resources.geometry(before.geometryId),
                Wkb()
            };
        }
        if (before.attributesId) {
            data.attributes = createAttributesDiff(
                resources.attributes(before.attributesId),
                Attributes()
            );
        }
        if (before.descriptionId) {
            data.description = DescriptionDiff {
                resources.description(before.descriptionId),
                Description()
            };
        }
        result.emplace(objectId,
            ObjectRevisionDiff(data,
                RevisionID(objectId, prevCommitId),
                RevisionID(objectId, commitId)));
    };

    forAllRecords(oidToRevDataDiff, onUpdate, onCreate, onDelete);
    return result;
}

std::optional<AttributesDiff>
createAttributesDiff(const Attributes& oldAttrs, const Attributes& newAttrs)
{
    AttributesDiff diff;
    for (const auto& p : oldAttrs) {
        const auto& attrKey = p.first;
        ASSERT(!attrKey.empty());

        auto it = newAttrs.find(attrKey);
        if (it == newAttrs.end()) {
            diff.insert({attrKey, StringDiff(p.second, EMPTY_STRING)});
        } else if (p.second != it->second) {
            diff.insert({attrKey, StringDiff(p.second, it->second)});
        }
    }
    for (const auto& p : newAttrs) {
        const auto& attrKey = p.first;
        ASSERT(!attrKey.empty());
        auto it = oldAttrs.find(attrKey);
        if (it == oldAttrs.end()) {
            diff.insert({attrKey, StringDiff(EMPTY_STRING, p.second)});
        }
    }
    if (!diff.empty()) {
        return diff;
    }
    return std::optional<AttributesDiff>();
}

} // namespace maps::wiki::revision
