#include "helpers.h"
#include "sql_strings.h"
#include <yandex/maps/wiki/revision/common.h>
#include <yandex/maps/wiki/revision/exception.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/common/include/digest.h>

namespace maps::wiki::revision::helpers {

CommitState
stringToCommitState(const std::string& str)
{
    REQUIRE(!str.empty(), "empty commit state");

    if (str == sql::val::COMMIT_STATE_APPROVED) {
        return CommitState::Approved;
    }
    if (str == sql::val::COMMIT_STATE_DRAFT) {
        return CommitState::Draft;
    }
    REQUIRE(false, "invalid commit state value: " << str);
}

const std::string&
commitStateToString(CommitState state)
{
    switch(state) {
        case CommitState::Approved:
            return sql::val::COMMIT_STATE_APPROVED;
        case CommitState::Draft:
            return sql::val::COMMIT_STATE_DRAFT;
    }
    REQUIRE(false, "invalid commit state value: "
        << static_cast<std::underlying_type_t<CommitState>>(state));
}

Hstore
attributesToHstore(
    const pqxx::transaction_base& work,
    const Attributes& attrs,
    Attributes* newAttrs)
{
    std::string result;
    for (const auto& attr : attrs) {
        if (!attr.first.empty() && !attr.second.empty()) {
            if (!result.empty()) {
                result.push_back(',');
            }
            result += (
                "['" + work.esc(attr.first) +
                "','" + work.esc(attr.second) + "']");
            if (newAttrs) {
                newAttrs->insert(attr);
            }
        }
    }
    if (result.empty()) {
        return Hstore();
    }
    return Hstore("hstore(ARRAY[" + result + "])");
}

Attributes
stringToAttributes(const std::string& attrs)
{
    Attributes result;
    if (attrs.empty()) {
        return result;
    }

    REQUIRE(attrs.size() >= 2, "invalid attributes: " << attrs);
    REQUIRE(*attrs.begin() == '{' && *attrs.rbegin() == '}', "invalid attributes: " << attrs);

    if (attrs.size() == 2) {
        return result;
    }

    std::string key;
    std::string value;
    auto insertData = [&] () {
        auto pair = result.emplace(key, value);
        REQUIRE(pair.second,
            "invalid attributes (duplicate key: " << key << "): " << attrs);
    };

    bool insideQuotes = false;

    size_t length = attrs.length() - 2;
    for (const char* str = attrs.c_str() + 1; length > 0; ++str, --length) {
        if (*str == '"') {
            insideQuotes = !insideQuotes;
            continue;
        }
        if (!insideQuotes) {
            if (*str == ' ' || *str == '\t') {
                continue;
            }
            if (*str == ',') {
                if (key.empty()) {
                    key.swap(value);
                    REQUIRE(!key.empty(), "invalid attributes (empty key): " << attrs);
                }
                else {
                    insertData();
                    key.clear();
                    value.clear();
                }
                continue;
            }
        }
        if (*str == '\\') {
            REQUIRE(length > 1, "invalid attributes (unexpectend escape at end): " << attrs);
            --length;
            ++str;
        }
        value += *str;
    }

    REQUIRE(!insideQuotes, "invalid attributes (unexpectend end of literal): " << attrs);
    REQUIRE(!key.empty(), "invalid attributes (odd number of parts, value missed): " << attrs);

    insertData();

    return result;
}

std::vector<DBID>
loadIds(const pqxx::result& r, const std::string& fieldName)
{
    std::vector<DBID> ids;
    if (!r.empty()) {
        ids.reserve(r.size());
        for (size_t i = 0; i < r.size(); ++i) {
            ids.push_back(r[i][fieldName].as<DBID>());
        }
    }
    return ids;
}

pqxx::result
lockData(
    pqxx::transaction_base& work, const std::string& query,
    const char* context)
{
    try {
        return work.exec(query);
    }
    catch (const pqxx::sql_error& ex) {
        if (strstr(ex.what(), "could not obtain lock") != 0) {
            throw ConcurrentLockDataException() <<
                "concurrent locking " << context << " : " << ex.what();
        }
        throw;
    }
}

DBID stringToDBID(const std::string& s)
{
    return std::stoull(s);
}

DBIDSet
stringToDBIDSet(const std::string& s) {
    DBIDSet result;
    if (!s.empty()) {
        std::string::size_type begin = 0;
        std::string::size_type end = s.find(',');
        for (; end != std::string::npos; end = s.find(',', begin)) {
            result.insert(stringToDBID(s.substr(begin, end - begin)));
            begin = end + 1;
        }
        result.insert(stringToDBID(s.substr(begin)));
    }
    return result;
}

void
checkObjectId(DBID objectId)
{
    REQUIRE(objectId != 0, "invalid object id");
}

void
checkCommitId(DBID commitId)
{
    REQUIRE(commitId != 0, "invalid commit id");
}

void
checkRevisionId(const RevisionID& revisionId)
{
    REQUIRE(revisionId.valid(), "invalid revision id: " << revisionId);
}

void
checkUserId(UserID uid)
{
    REQUIRE(uid, "invalid user id");
}

void
checkLimit(size_t limit)
{
    REQUIRE(limit, "zero limit value");
}

void
checkRevisionIds(const ConstRange<RevisionID>& ids)
{
    auto itc = ids.iterator();
    for (const RevisionID* idp = itc->next(); idp != nullptr; idp = itc->next()) {
        checkRevisionId(*idp);
    }
}

RevisionIds
revisionIdsToVector(const ConstRange<RevisionID>& revisionIds)
{
    RevisionIds result;
    auto itc = revisionIds.iterator();
    for (const auto* idp = itc->next(); idp != nullptr; idp = itc->next()) {
        checkRevisionId(*idp);
        result.push_back(*idp);
    }
    return result;
}

} // namespace maps::wiki::revision::helpers
