#pragma once

#include <yandex/maps/wiki/revision/common.h>

#include <memory>

namespace maps::wiki::revision {

class BranchManager;
class BranchData;

class Branch {
public:
    using LockId = uint64_t;
    static const LockId MAX_LOCK_ID;

    Branch(Branch&& other) noexcept;
    Branch(const Branch& other);
    ~Branch();

    Branch& operator = (const Branch& other);

    DBID id() const;

    BranchType type() const;
    BranchState state() const;


    static bool isReadOnlyType(const BranchType& type);
    static bool isWriteAllowedType(const BranchType& type);

    bool isWritingAllowed() const;
    bool isReadingAllowed() const;

    UserID createdBy() const;
    const std::string& createdAt() const;

    UserID finishedBy() const;
    const std::string& finishedAt() const;

    const Attributes& attributes() const;

    bool concatenateAttributes(
            pqxx::transaction_base& work, const Attributes& attributes);

    bool setType(pqxx::transaction_base& work, BranchType type);
    bool setState(pqxx::transaction_base& work, BranchState state);

    bool finish(pqxx::transaction_base& work, UserID uid);

    // for Approved and Stable
    bool touchCreated(pqxx::transaction_base& work, UserID uid);


    /**
     * Available lock ids are [0 .. MAX_LOCK_ID]
     * @throw maps::wiki::revision::BranchLockIdOutOfRange
     *   if lock id is out of range [0 .. MAX_LOCK_ID]

     * Lock ids in use in MAPSPRO project:
     * 0: "global lock" required for branch editing operations and editor tool
     * 1: lock required for update of revision_meta
     * 2: lock required for editor tool
     * 3: lock required for update of geometry locks table
     */

    enum class LockType { Shared, Exclusive };

    bool tryLock(pqxx::transaction_base& work, LockId lockId) const;

    bool tryLock(pqxx::transaction_base& work, LockId lockId, LockType type) const;

    enum class LockMode { Wait, Nowait };

    /**
     * @throw maps::wiki::revision::BranchAlreadyLockedException
     *   if mode == LockMode::Nowait and branch is already locked
     */
    void lock(pqxx::transaction_base& work, LockId lockId, LockMode mode) const;

    /**
     * @throw maps::wiki::revision::BranchAlreadyLockedException
     *   if mode == LockMode::Nowait and lock could not be granted
     */
    void lock(
            pqxx::transaction_base& work,
            LockId lockId, LockMode mode, LockType type) const;

private:
    friend class BranchManager;
    explicit Branch(const BranchData& data);

    class Impl;
    std::unique_ptr<Impl> impl_;
};

} // namespace maps::wiki::revision
