#pragma once

#include <maps/wikimap/mapspro/libs/types/include/uid.h>

#include <pqxx/pqxx>

#include <cstdint>
#include <cstddef>
#include <list>
#include <map>
#include <optional>
#include <string>
#include <set>

namespace maps::wiki::revision {

using DBID = uint64_t;
using UserID = types::TUid;

constexpr DBID TRUNK_BRANCH_ID = 0;

using Wkb = std::string;
using Hstore = std::string;
using Description = std::string;
using DBIDSet = std::set<DBID>;
using Attributes = std::map<std::string, std::string>;
using ObjectIdRange = std::pair<DBID, DBID>; // [minOid, maxOid]
using OptionalDBID = std::optional<DBID>;

enum class DescriptionLoadingMode {
    Load,   // Loads description for non-relation objects (for backward compartibility)
    Skip,   // Do not load description
    LoadAll // Loads description for all types of objects
};

enum class CommitState {
    Draft,
    Approved = 2
};
std::ostream& operator << (std::ostream& os, CommitState commitState);

enum class RevisionType {
    RegularObject,
    Relation
};

enum class BranchType
{
    //general-purpose branch for inserting new revisions
    Trunk,
    //moderated meta-branch
    //(revisions aren't ordered by commitId here)
    Approved,
    //separated branch for release purposes
    Stable,
    //same as Stable from library side of view
    Archive,
    //same as Stable, but isn't available for writing
    Deleted
};
std::istream& operator >> (std::istream& is, BranchType& type);
std::ostream& operator << (std::ostream& os, BranchType type);

enum class BranchState { Unavailable, Normal, Progress };
std::istream& operator >> (std::istream& is, BranchState& state);
std::ostream& operator << (std::ostream& os, BranchState state);


class RelationData {
public:
    RelationData(DBID masterObjectId, DBID slaveObjectId);

    bool isRelation()    const { return slaveObjectId_ != 0; }

    DBID masterObjectId() const { return masterObjectId_; }
    DBID slaveObjectId()  const { return slaveObjectId_; }

    bool operator == (const RelationData& data) const
    {
        return masterObjectId_ == data.masterObjectId_ &&
                slaveObjectId_ == data.slaveObjectId_;
    }

    bool operator != (const RelationData& data) const
    {
        return masterObjectId_ != data.masterObjectId_ ||
                slaveObjectId_ != data.slaveObjectId_;
    }

    bool operator < (const RelationData& data) const
    {
        if (masterObjectId_ < data.masterObjectId_) {
            return true;
        }
        if (masterObjectId_ > data.masterObjectId_) {
            return false;
        }
        return slaveObjectId_ < data.slaveObjectId_;
    }

private:
    DBID masterObjectId_;
    DBID slaveObjectId_;
};


struct ObjectData {
    ObjectData() = default;

    explicit ObjectData(
            std::optional<Attributes> attributes,
            std::optional<Description> description = std::nullopt,
            std::optional<Wkb> geometry = std::nullopt,
            std::optional<RelationData> relationData = std::nullopt,
            bool deleted = false)
        : attributes(std::move(attributes))
        , description(std::move(description))
        , geometry(std::move(geometry))
        , relationData(relationData)
        , deleted(deleted)
    {}

    RevisionType revisionType() const
    {
        return !relationData
            ? RevisionType::RegularObject
            : RevisionType::Relation;
    }

    std::optional<Attributes> attributes;
    std::optional<Description> description;
    std::optional<Wkb> geometry;
    std::optional<RelationData> relationData;
    bool deleted{false};
};

} // namespace maps::wiki::revision
