#pragma once

#include <boost/noncopyable.hpp>
#include <initializer_list>
#include <memory>
#include <type_traits>

namespace maps::wiki::revision {


template <typename T>
class ConstIteratorRangeData : public boost::noncopyable {
public:
    virtual ~ConstIteratorRangeData() = default;

    using IteratorPtr = std::unique_ptr<ConstIteratorRangeData<T>>;
    using Pointer = const T *;

    virtual bool empty() const = 0;
    virtual Pointer next() = 0;

    virtual IteratorPtr clone() const = 0;
};


template <typename T, typename Iter, typename Base = ConstIteratorRangeData<T> >
class IteratorRangeDataImpl : public Base {
public:
    IteratorRangeDataImpl(Iter begin, Iter end) : begin_(begin), end_(end) {}

    virtual bool empty() const { return begin_ == end_; }

    virtual typename Base::Pointer next()
    {
        if (begin_ == end_) {
            return 0;
        }
        return &(*begin_++);
    }

protected:
    using IteratorPtr = typename Base::IteratorPtr;

    virtual IteratorPtr clone() const
    {
        return IteratorPtr(new IteratorRangeDataImpl<T, Iter, Base>(begin_, end_));
    }

private:
    Iter begin_;
    Iter end_;
};


template <typename T>
class ConstRange : public boost::noncopyable {
protected:
    using ConstIterData = ConstIteratorRangeData<T>;

public:
    template <typename Iter>
    ConstRange(Iter begin, Iter end)
        : constData_(new IteratorRangeDataImpl<T, Iter>(begin, end))
    {
    }

    template <typename Cont,
        typename = typename Cont::const_iterator,
        typename = std::enable_if_t<std::is_convertible_v<typename Cont::value_type*, T*>>>
    ConstRange(const Cont& cont)
    {
        using Iter = typename Cont::const_iterator;

        constData_.reset(new IteratorRangeDataImpl<T, Iter>(cont.begin(), cont.end()));
    }

    ConstRange(std::initializer_list<T> list)
    {
        using Iter = typename std::initializer_list<T>::const_iterator;

        constData_.reset(new IteratorRangeDataImpl<T, Iter>(list.begin(), list.end()));
    }

    using ConstIterator = typename ConstIterData::IteratorPtr;
    using Iterator = typename ConstIterData::IteratorPtr;

    bool empty() const { return constData_->empty(); }

    ConstIterator constIterator() const { return constData_->clone(); }
    ConstIterator iterator() const { return constData_->clone(); }

private:
    ConstIterator constData_;
};

} // namespace maps::wiki::revision
