#pragma once

#include <yandex/maps/wiki/revision/common.h>
#include <yandex/maps/wiki/revision/commit.h>
#include <yandex/maps/wiki/revision/diff.h>
#include <yandex/maps/wiki/revision/objectrevision.h>
#include <yandex/maps/wiki/revision/range.h>
#include <yandex/maps/wiki/revision/reader.h>
#include <yandex/maps/wiki/revision/snapshot.h>
#include <yandex/maps/wiki/revision/historical_snapshot.h>

#include <boost/noncopyable.hpp>

#include <map>
#include <vector>
#include <memory>
#include <list>

namespace maps::wiki::revision {

using Transaction = pqxx::transaction_base;

class Branch;

class RevisionsGateway : boost::noncopyable {
public:
    explicit RevisionsGateway(Transaction& work); // branch == Trunk

    RevisionsGateway(Transaction& work, const Branch& branch);

    RevisionsGateway(RevisionsGateway&&) noexcept;
    RevisionsGateway& operator=(RevisionsGateway&&) noexcept;

    ~RevisionsGateway();

    Transaction& work() const;

    DBID branchId() const;

    BranchType branchType() const;

    /// returns ID with empty commitId
    RevisionID acquireObjectId() const;

    std::list<ObjectIdRange> acquireObjectIds(size_t number) const;

    /// returns current sequence value for ObjectId
    DBID lastObjectId() const;

    /// VERY DANGER METHOD. USE WITH CAUTION
    /// increase object id sequence by the number
    /// returns the last acquired object id
    DBID reserveObjectIds(size_t number) const;


    /// Interface to read ID's and revisions directly
    Reader reader() const;

    // Interface to read the most recent revision in gateway's branch
    // Generally speaking, such snapshot is not
    Snapshot snapshot(DBID commitId) const;

    // Same, except for the approved branch: this function will load the specified commit
    // from database, creating snapshot from loaded commit's approveOrder
    Snapshot stableSnapshot(DBID commitId) const;

    // Second version of snapshot creation.
    // Won't issue additional database requests
    Snapshot snapshot(const SnapshotId& id) const;

    // Interface to read all ID's and revisions
    // limited by minCommitId/maxCommitId (can be 0)
    HistoricalSnapshot historicalSnapshot(DBID maxCommitId) const;
    HistoricalSnapshot historicalSnapshot(DBID minCommitId, DBID maxCommitId) const;


    // Commit operations
    // returns 0 for empty data
    DBID headCommitId() const;

    // Computes maximum snapshot limiting parameter,
    // based on current branch.
    // Returns
    //      maxCommitId for trunk and stable branches,
    //      maxApproveOrder for approved meta-branch
    // returns 0 for empty data
    SnapshotId maxSnapshotId() const;

    using NewRevisionData = std::pair<RevisionID, ObjectRevision::Data>;

    Commit createCommit(
        const ConstRange<NewRevisionData>& newData, UserID createdBy, const Attributes& attributes) const;

    template <typename Iter>
    Commit createCommit(
        Iter newDataBegin, Iter newDataEnd, UserID createdBy, const Attributes& attributes) const
    {
        const ConstRange<NewRevisionData> range(newDataBegin, newDataEnd);
        return createCommit(range, createdBy, attributes);
    }

    /// check recently and existed revisions
    void checkConflicts(const ConstRange<RevisionID>& ids) const;

    // truncates all tables, restarts sequences
    void truncateAll() const;

    // create the following branches in revision.branch table:
    //      * BranchType::Trunk
    void createDefaultBranches() const;

private:
    class Impl;
    std::unique_ptr<Impl> impl_;
};


} // namespace maps::wiki::revision
