#include "query_generator.h"
#include "sql_strings.h"
#include "helpers.h"
#include <maps/libs/common/include/exception.h>

#include <sstream>

namespace maps::wiki::revision {

using namespace helpers;

namespace {

using IdsRange = std::pair<DBID, DBID>; // [min, end), example: [1,3) -> range size = 2
using IdsRanges = std::vector<IdsRange>;

const size_t NORMALIZE_IDS_MIN_SIZE = 100;
const size_t NORMALIZE_IDS_MIN_RANGE_SIZE = 50; // (xx0 .. xx101)

inline void
updateNormalizeResult(IdsRanges& idsRanges, DBIDSet& ids, const IdsRange& idsRange)
{
    if (idsRange.first + NORMALIZE_IDS_MIN_RANGE_SIZE <= idsRange.second) {
        idsRanges.push_back(idsRange);
        return;
    }

    for (DBID id = idsRange.first; id < idsRange.second; ++id) {
        ids.insert(id);
    }
}

std::pair<IdsRanges, DBIDSet>
normalizeIds(const DBIDSet& ids)
{
    ASSERT(ids.size() >= NORMALIZE_IDS_MIN_SIZE);

    const DBID firstId = *ids.begin();
    ASSERT(firstId > 0);

    std::pair<IdsRanges, DBIDSet> result;
    IdsRange curIdsRange(firstId, firstId + 1);

    for (DBIDSet::const_iterator it = ids.begin(); ++it != ids.end(); ) {
        if (curIdsRange.second != *it) {
            updateNormalizeResult(result.first, result.second, curIdsRange);
            curIdsRange.first = *it;
        }
        curIdsRange.second = *it + 1;
    }

    updateNormalizeResult(result.first, result.second, curIdsRange);
    return result;
}

std::string
buildFilterByAttrValuesInternal(
    const std::string& fieldName,
    const DBIDSet& ids,
    bool checkRanges = true)
{
    ASSERT(!ids.empty());

    size_t size = ids.size();
    if (size == 1) {
        return fieldName + "=" + std::to_string(*ids.begin());
    }

    if (*ids.begin() + size == *ids.rbegin() + 1) {
        std::stringstream res;
        res << '('     << fieldName << '>' << (*ids.begin()) - 1
            << " AND " << fieldName << '<' << (*ids.rbegin()) + 1 << ')';
        return res.str();
    }

    if (checkRanges && ids.size() >= NORMALIZE_IDS_MIN_SIZE) {
        auto result = normalizeIds(ids);
        if (!result.first.empty()) {
            std::stringstream res;
            res << '(';
            bool first = true;
            for (const auto& idsRange : result.first) {
                if (first) {
                    first = false;
                } else {
                    res << " OR ";
                }
                res << '('     << fieldName << '>' << idsRange.first - 1
                    << " AND " << fieldName << '<' << idsRange.second << ')';
            }
            if (!result.second.empty()) {
                res << " OR "
                    << buildFilterByAttrValuesInternal(fieldName, result.second, false);
            }
            res << ')';
            return res.str();
        }
    }
    return "(" + fieldName + ">" + std::to_string(*ids.begin() - 1) + " AND " +
                 fieldName + "<" + std::to_string(*ids.rbegin() + 1) + " AND " +
                 fieldName + " IN (" + valuesToString(ids) + "))";
}

} // namespace

std::string
QueryGenerator::buildFilterByAttrValues(const std::string& fieldName, const DBIDSet& ids)
{
    ASSERT(!fieldName.empty());
    ASSERT(!ids.empty());

    return buildFilterByAttrValuesInternal(fieldName, ids);
}

std::string
QueryGenerator::buildSpecialFilter(const CommitIdToObjectIds& commit2objects,
    const std::string& commitFieldName)
{
    ASSERT(!commit2objects.empty());

    std::stringstream ss;
    if (commit2objects.size() != 1) {
        ss << '(';
    }

    bool first = true;
    for (const auto& p : commit2objects) {
        if (!first) {
            ss << " OR ";
        }
        else {
            first = false;
        }
        ss << '(' << commitFieldName << '=' << p.first
           << " AND " << buildFilterByAttrValuesInternal(sql::col::OBJECT_ID, p.second) << ')';
    }

    if (commit2objects.size() != 1) {
        ss << ')';
    }

    return ss.str();
}

std::pair<CommitIdToObjectIds, size_t>
QueryGenerator::buildCommitIdToObjectIds(const ConstRange<ObjectRevision::ID>& ids)
{
    CommitIdToObjectIds commit2objectIds;
    size_t size = 0;

    auto itc = ids.iterator();
    for (const ObjectRevision::ID* idp = itc->next(); idp != 0; idp = itc->next()) {
        if (commit2objectIds[idp->commitId()].insert(idp->objectId()).second) {
            ++size;
        }
    }
    return std::make_pair(commit2objectIds, size);
}

size_t QueryGenerator::countRows(const CommitIdToObjectIds& commits)
{
    size_t counter = 0;
    for (const auto& pair: commits) {
        counter += pair.second.size();
    }
    return counter;
}


} // namespace maps::wiki::revision
