#include "reader_impl.h"
#include "result_interpreter.h"
#include "query_generator.h"
#include "filter_generators.h"
#include "context.h"
#include "revision_data.h"
#include "sql_strings.h"
#include <yandex/maps/wiki/revision/filters.h>

#include <boost/algorithm/string/join.hpp>
#include <boost/lexical_cast.hpp>

#include <cstddef>
#include <memory>
#include <sstream>
#include <unordered_set>
#include <unordered_map>


namespace maps::wiki::revision {

namespace {

//When loading RevisionIds via pqxx::pipeline,
//split input data into batches of the given size
const size_t LOAD_REVISION_IDS_BATCH_SIZE = 1000;

//When loading Revisions via pqxx::pipeline,
//split input data into batches of the given size
const size_t LOAD_REVISIONS_BATCH_SIZE = 500;

//When loading anything via pqxx::pipeline,
//don't issue all the requests at once
//(since this will lead to a huge memory usage).
//Instead split the batches array into a batches of the given size
const size_t MAX_NUMBER_OF_REQUESTS_IN_PIPELINE = 10;

std::string makeSelectClause(
    const db::Fields& request
)
{
    std::ostringstream query;
    bool first = true;
    for (auto field: request) {
        if (first) {
            query << " SELECT ";
            first = false;
        } else {
            query << ", ";
        }

        OptionalString transformFunc;
        OptionalString columnAlias;
        OptionalString tableAlias;
        std::string column;

        switch (field) {
            case db::Field::AttributesId:
            case db::Field::CommitId:
            case db::Field::DescriptionId:
            case db::Field::GeometryId:
            case db::Field::IsDeleted:
            case db::Field::MasterObjectId:
            case db::Field::NextCommitId:
            case db::Field::ObjectId:
            case db::Field::PrevCommitId:
            case db::Field::SlaveObjectId:
                tableAlias = sql::alias::OBJECT_REVISION;
                column = boost::lexical_cast<std::string>(field);
                break;

            case db::Field::Attributes:
                tableAlias = sql::alias::ATTRIBUTES;
                column = sql::col::CONTENTS;
                transformFunc = sql::func::HSTORE_TO_ARRAY;
                columnAlias = sql::col::ATTRIBUTES;
                break;
        }

        query << sql::makeColumnExpr(
            column,
            tableAlias,
            transformFunc,
            columnAlias
        );
    }

    return query.str();
}


std::string makeFromClause(
    const db::Partitions& partitions,
    const db::Tables& tables
)
{
    auto joinTable = [&](
        const std::string& table,
        const std::string& alias,
        const std::string& joinColumn)
    {
        return
            " JOIN " + table + " " + alias +
            " ON (" +
                sql::alias::OBJECT_REVISION + "." + joinColumn + " = " +
                alias + "." + sql::col::ID + ") ";
    };

    std::string query = " FROM ";

    bool isPartRelation = false;
    if (partitions.size() == 1) {
        const auto& partition = *partitions.begin();
        switch (partition) {
            case db::Partition::RegularWithGeometry:
                query += sql::table::OBJECT_REVISION_WITH_GEOMETRY;
                break;
            case db::Partition::RegularWithoutGeometry:
                query += sql::table::OBJECT_REVISION_WITHOUT_GEOMETRY;
                break;
            case db::Partition::Relation:
                isPartRelation = true;
                query += sql::table::OBJECT_REVISION_RELATION;
                break;
        }
    } else {
        query += sql::table::OBJECT_REVISION;
    }

    query += " " + sql::alias::OBJECT_REVISION;

    query += joinTable(
        sql::table::COMMIT,
        sql::alias::COMMIT,
        sql::col::COMMIT_ID
    );
    if (tables.count(db::Table::Attributes) > 0) {
        query += " LEFT";
        query += joinTable(
            isPartRelation
                ? sql::table::ATTRIBUTES_RELATIONS
                : sql::table::ATTRIBUTES,
            sql::alias::ATTRIBUTES,
            sql::col::ATTRIBUTES_ID
        );
    }
    return query;
}


template<typename Container>
void batchLoadData(
    pqxx::transaction_base& txn,
    const Container& inputData,
    size_t batchSize,
    std::function<
        OptionalString(const maps::common::Batch<Container>&)
    > queryGenerator,
    std::function<
        void(
            const pqxx::result&,
            const maps::common::Batch<Container>&
        )
    > resultInterpreter
)
{
    if (inputData.empty()) {
        return;
    }

    using Batch = maps::common::Batch<Container>;

    //storing these batches is required for performing additional checks in resultInterpreter
    std::map<pqxx::pipeline::query_id, Batch> queriedBatches;

    auto loadAllFromPipeline = [&] (pqxx::pipeline& pipeline)
    {
        while (!pipeline.empty()) {
            const auto& retrievedResult = pipeline.retrieve();
            resultInterpreter(
                retrievedResult.second,
                queriedBatches.at(retrievedResult.first)
            );
        }
    };

    pqxx::pipeline pipeline(txn);

    size_t batchIndex = 0;
    for (const auto& batch: common::makeBatches(inputData, batchSize)) {
        if ((batchIndex % MAX_NUMBER_OF_REQUESTS_IN_PIPELINE) == 0) {
            loadAllFromPipeline(pipeline);
        }

        auto query = queryGenerator(batch);
        if (query) {
            auto queryId = pipeline.insert(*query);
            queriedBatches.insert({queryId, batch});
        }
    }
    //fetching last results
    loadAllFromPipeline(pipeline);
}

} //anonymous namespace

ReaderImpl::ReaderImpl(
    pqxx::transaction_base& txn,
    BranchType branchType,
    DBID branchId,
    DescriptionLoadingMode descriptionLoadingMode
)
    : branchType_(branchType)
    , branchId_(branchId)
    , txn_(txn)
    , descriptionLoadingMode_(descriptionLoadingMode)
{
    REQUIRE(
        branchType_ == BranchType::Trunk ||
        branchId_ != TRUNK_BRANCH_ID,
        "Wrong combination of branchType " << branchType_ <<
            " and branchId " << branchId_
    );
}

std::string ReaderImpl::makeWhereClause(
    const std::string& filterStatement,
    LoadLimitations loadLimitations,
    const OptionalSnapshotId& snapshotId
) const
{
    std::ostringstream query;
    std::vector<std::string> clauses{
        filterStatement,
        makeCommitStateFilterClause(loadLimitations),
        makeHistoricalFilterClause(loadLimitations, snapshotId)
    };

    bool first = true;
    for (const auto& clause: clauses) {
        if (clause.empty()) {
            continue;
        }

        if (first) {
            query << " WHERE ";
        } else {
            query << " AND ";
        }
        query << "(" << clause << ")";
        first = false;
    }

    return query.str();
}

std::string ReaderImpl::makePartitionQuery(
    const db::StructuredSubquery& subquery,
    LoadLimitations loadLimitations,
    const OptionalSnapshotId& snapshotId
) const
{
    std::ostringstream query;
    query << makeSelectClause(subquery.selectList);
    query << makeFromClause(subquery.partitions, subquery.joinList);

    query << makeWhereClause(
        subquery.filterStatement,
        loadLimitations,
        snapshotId
    );

    return query.str();
}

std::string ReaderImpl::makeQuery(
    const db::StructuredQuery& strQuery,
    LoadLimitations limitations,
    const OptionalSnapshotId& snapshotId,
    const OptionalSize& limit
) const
{
    REQUIRE(
        !strQuery.subqueries.empty(),
        "Empty queries aren't allowed"
    );
    std::ostringstream query;

    bool first = true;
    for (const auto& subquery: strQuery.subqueries) {
        if (first) {
            first = false;
        } else {
            query << " UNION ";
        }
        query << makePartitionQuery(
            subquery,
            limitations,
            snapshotId
        );
    }

    if (limit && !needExcludeHistoricals(limitations)) {
        query << " LIMIT " << *limit;
    }

    return query.str();
}

//TODO: rewrite using filters after filters refactoring
std::string ReaderImpl::makeCommitStateFilterClause(LoadLimitations loadLimitations) const
{
    std::ostringstream query;
    //Three cases are possible:
    //1. Reading from Trunk branch:
    //   Commits should have 'trunk' flag and must not be in 'revoked' state
    //2. Reading from Approved meta-branch:
    //   Commits should have 'trunk' flags and must be in 'approved' state
    //3. Reading from Stable branch with id brachId:
    //   (
    //       Non-revoked commits with stable_branch_id = branchId and
    //       Non-revoked trunk commits with stable_branch_id < branchId
    //   )
    //   should be considered
    //
    //Historical revisions could not be filtered when requesting data from stable branch
    //We'll filter data afterwards.
    if (loadLimitations == LoadLimitations::None) {
        return query.str();
    }

    if (branchType_ == BranchType::Trunk) {
        query <<
            sql::alias::COMMIT << "." << sql::col::IS_TRUNK;
    } else if (branchType_ == BranchType::Approved) {
        query <<
            sql::alias::COMMIT << "." << sql::col::IS_TRUNK << " AND " <<
            sql::alias::COMMIT << "." << sql::col::APPROVE_ORDER << " > 0";
    } else {
        //handling stable branch
        query << "(" <<
            "(" <<
                sql::alias::COMMIT << "." << sql::col::STABLE_BRANCH_ID << " = " << branchId_ <<
            ") OR (" <<
                sql::alias::COMMIT << "." << sql::col::IS_TRUNK << " AND " <<
                sql::alias::COMMIT << "." << sql::col::STABLE_BRANCH_ID << " < " << branchId_ <<
            ")" <<
        ")";
    }

    return query.str();
}


//TODO: rewrite using filters after filters refactoring
std::string ReaderImpl::makeHistoricalFilterClause(
    LoadLimitations loadLimitations,
    const OptionalSnapshotId& snapshotId
) const
{
    std::vector<std::string> clauses;

    //When snapshotId specified, both commit.id and object_revision.commit_id
    //should not exceed snapshotId
    if (snapshotId) {
        std::ostringstream query;
        if (
            (branchType_ == BranchType::Approved) &&
            snapshotId->hasApproveOrder()
        ) {
            query << "(" <<
                sql::alias::COMMIT << "." << sql::col::APPROVE_ORDER << " <= " <<
                    snapshotId->approveOrder() <<
            ")";
        } else {
            query << "(" <<
                sql::alias::COMMIT << "." << sql::col::ID << " <= " <<
                    snapshotId->commitId() << " AND " <<
                sql::alias::OBJECT_REVISION << "." << sql::col::COMMIT_ID << " <= " <<
                    snapshotId->commitId() <<
            ")";
        }
        clauses.emplace_back(query.str());
    }

    //While filtering past (historical) revisions, multiple cases are possible:
    //1. Reading from Trunk branch:
    //   next_commit_id should be zero or
    //   next_commit_id should exceed snapshotId if it's specified
    //2. Reading from Stable and Approved branches is more complicaed:
    //   If no filters were applied, we can simply take the revision with maximum commitId.
    //   (this works since due to the following expression:
    //      (
    //          (object_id1 == object_id2) &&
    //          (approve_order1 > 0) &&
    //          (approve_order2 > 0)
    //      )
    //          =>
    //      (
    //          (approve_order1 < approve_order2) <=>
    //          (commit_id1 < commit_id2)
    //      )
    //   If user-generated filter is applied, there is NO way of correct excluding of past revisions
    //   (
    //      i. e. user can add filters::ObjRevAttr::isNotDeleted() filter,
    //      and we'll receive non-head revision if object is marked as deleted on the HEAD
    //   )
    if (
        (loadLimitations == LoadLimitations::Snapshot) &&
        (branchType_ == BranchType::Trunk)
    ) {
        std::ostringstream query;
        query << "(" <<
            sql::alias::OBJECT_REVISION << "." << sql::col::NEXT_COMMIT_ID << " = 0";
        if (snapshotId) {
            query << " OR " <<
                sql::alias::OBJECT_REVISION << "." << sql::col::NEXT_COMMIT_ID <<
                " > " << snapshotId->commitId();
        }
        query << ")";
        clauses.emplace_back(query.str());
    }

    return boost::join(
        clauses,
        " AND "
    );
}


//filter-aware version of loading revisions
Revisions ReaderImpl::loadRevisions(
    LoadLimitations loadLimitations,
    const OptionalSnapshotId& snapshotId,
    const OptionalSize& limit,
    const filters::FilterExpr& expr
) const
{
    auto strQuery = db::StructuredQueryProvider::revision(
        txn_.conn(),
        expr
    );
    if (strQuery.subqueries.empty()) {
        return {};
    }

    auto query = makeQuery(strQuery, loadLimitations, snapshotId, limit);

    const auto& pqxxResult = txn_.exec(query);
    auto revisions = loadRevisionsFromPqxx(pqxxResult);

    revisions.sort();
    if (needExcludeHistoricals(loadLimitations)) {
        //performing filtering via additional request
        revisions = removeNonHeadRevisions(std::move(revisions), snapshotId);
    }
    fetchHeavyContent(revisions);
    return revisions;
}

//pipelined version of loading revisions
Revisions ReaderImpl::loadRevisions(
    LoadLimitations loadLimitations,
    const OptionalSnapshotId& snapshotId,
    const OptionalSize& limit,
    const DBIDSet& objectIds
) const
{
    using Batch = maps::common::Batch<DBIDSet>;

    auto queryGenerator = [&] (const Batch& batch) -> OptionalString
    {
        auto batchFilter = filters::ObjRevAttr::objectId().in(batch);

        auto query = db::StructuredQueryProvider::revision(
            txn_.conn(),
            batchFilter
        );
        if (query.subqueries.empty()) {
            return std::nullopt;
        }

        return makeQuery(
            query, loadLimitations, snapshotId, limit
        );
    };

    Revisions revisions;
    auto resultInterpreter = [&](
        const pqxx::result& pqxxResult,
        const Batch& /* batch */
    )
    {
        Revisions batchRevisions = loadRevisionsFromPqxx(pqxxResult);
        revisions.splice(
            revisions.end(),
            std::move(batchRevisions)
        );
    };

    batchLoadData<DBIDSet>(
        txn_,
        objectIds,
        LOAD_REVISIONS_BATCH_SIZE,
        queryGenerator,
        resultInterpreter
    );

    revisions.sort();
    if (needExcludeHistoricals(loadLimitations)) {
        //performing in-memory filtering
        revisions = excludeNonHeadRevisions(std::move(revisions));
    }
    fetchHeavyContent(revisions);
    return revisions;
}


/**
 * Some client classes (Snapshot, HistoricalSnapshot, Reader) need
 * support for both filtering and batched filtering
 * by a custom set of columns.
 *
 * NOTE: batching will be performed without additional batchData modification.
 *
 * NOTE: When basicFilter is nullptr,
 *       and checkResultSize is Yes
 *       an extra check will be applied:
 *       Each revision specified in batchData should be loaded.
 */
template<typename Container>
Revisions ReaderImpl::loadRevisions(
    LoadLimitations loadLimitations,
    const OptionalSnapshotId& snapshotId,
    const OptionalSize& limit,
    const filters::FilterExpr* basicFilter,
    CheckResultSize checkResultSize,
    typename BatchFilterGenerator<Container>::type batchFilterGenerator,
    const Container& batchData
) const
{
    using Batch = maps::common::Batch<Container>;
    auto queryGenerator = [&] (const Batch& batch) -> OptionalString
    {
        auto batchFilter = basicFilter ?
            (
                batchFilterGenerator(batch) &&
                filters::ProxyReferenceFilterExpr(*basicFilter)
            ) :
            batchFilterGenerator(batch);

        auto query = db::StructuredQueryProvider::revision(
            txn_.conn(),
            batchFilter
        );
        if (query.subqueries.empty()) {
            return std::nullopt;
        }

        return makeQuery(
            query,
            loadLimitations,
            snapshotId,
            limit
        );
    };

    Revisions revisions;
    auto resultInterpreter = [&](
        const pqxx::result& pqxxResult,
        const Batch& batch
    )
    {
        if (!basicFilter && (checkResultSize == CheckResultSize::Yes)) {
            size_t batchSize = std::distance(batch.begin(), batch.end());
            REQUIRE(
                pqxxResult.size() == batchSize,
                "Some of requested objects failed to load. " <<
                "Expected size: " << batchSize << ", " <<
                "received size: " << pqxxResult.size()
            );
        }
        Revisions batchRevisions = loadRevisionsFromPqxx(pqxxResult);
        revisions.splice(
            revisions.end(),
            std::move(batchRevisions)
        );
    };

    batchLoadData<Container>(
        txn_,
        batchData,
        LOAD_REVISIONS_BATCH_SIZE,
        queryGenerator,
        resultInterpreter
    );

    revisions.sort();
    if (needExcludeHistoricals(loadLimitations)) {
        //performing filtering via additional request
        revisions = removeNonHeadRevisions(std::move(revisions), snapshotId);
    }
    fetchHeavyContent(revisions);
    return revisions;
}

void
ReaderImpl::fetchHeavyContent(Revisions& revisions) const
{
    using Ids = std::unordered_set<DBID>;
    using Batch = maps::common::Batch<Ids>;

    Ids geomIds;
    Ids descrIds;
    for (const auto& rev : revisions) {
        const auto& data = rev.revisionData();
        if (data.geometryId) {
            geomIds.insert(data.geometryId);
        }
        if (data.descriptionId
                && descriptionLoadingMode_ != DescriptionLoadingMode::Skip) {
            descrIds.insert(data.descriptionId);
        }
    }

    if (geomIds.empty() && descrIds.empty()) {
        return;
    }

    std::unordered_map<DBID, Wkb> wkbs;
    wkbs.reserve(geomIds.size());

    std::unordered_map<DBID, Description> descriptions;
    descriptions.reserve(descrIds.size());

    if (!geomIds.empty()) {
        auto loadBatchData = [&](const Batch& batchIds)
        {
            auto query =
                "SELECT " + sql::col::ID + ", " +
                            sql::func::GEOM_TO_BINARY + "(" + sql::col::CONTENTS + ")"
                " FROM " + sql::table::GEOMETRY +
                " WHERE " +
                    QueryGenerator::buildFilterByAttrValues(
                        sql::col::ID,
                        {batchIds.begin(), batchIds.end()});
            for (const auto& row : txn_.exec(query)) {
                wkbs[row[0].as<DBID>()] = pqxx::binarystring(row[1]).str();
            }
        };
        for (const auto& batch: common::makeBatches(geomIds, LOAD_REVISIONS_BATCH_SIZE)) {
            loadBatchData(batch);
        }
        REQUIRE(wkbs.size() == geomIds.size(), "geometries not loaded");
    }
    if (!descrIds.empty()) {
        auto loadBatchData = [&](const Batch& batchIds)
        {
            auto query =
                "SELECT " + sql::col::ID + ", " + sql::col::CONTENTS +
                " FROM " + sql::table::DESCRIPTION +
                " WHERE " +
                    QueryGenerator::buildFilterByAttrValues(
                        sql::col::ID,
                        {batchIds.begin(), batchIds.end()});
            for (const auto& row : txn_.exec(query)) {
                descriptions[row[0].as<DBID>()] = row[1].c_str();
            }
        };
        for (const auto& batch: common::makeBatches(descrIds, LOAD_REVISIONS_BATCH_SIZE)) {
            loadBatchData(batch);
        }
        REQUIRE(descriptions.size() == descrIds.size(), "descriptions not loaded");
    }

    for (auto& rev : revisions) {
        const auto& data = rev.revisionData();
        if (data.geometryId) {
            rev.setGeometry(wkbs.at(data.geometryId));
        }
        if (data.descriptionId
                && descriptionLoadingMode_ != DescriptionLoadingMode::Skip) {
            rev.setDescription(descriptions.at(data.descriptionId));
        }
    }
}

//explicit template instatiation of the above method
template Revisions ReaderImpl::loadRevisions<DBIDSet>(
    LoadLimitations loadLimitations,
    const OptionalSnapshotId& snapshotId,
    const OptionalSize& limit,
    const filters::FilterExpr* basicFilter,
    CheckResultSize checkResultSize,
    typename BatchFilterGenerator<DBIDSet>::type batchFilterGenerator,
    const DBIDSet& batchData
) const;

template Revisions ReaderImpl::loadRevisions<RevisionIds>(
    LoadLimitations loadLimitations,
    const OptionalSnapshotId& snapshotId,
    const OptionalSize& limit,
    const filters::FilterExpr* basicFilter,
    CheckResultSize checkResultSize,
    typename BatchFilterGenerator<RevisionIds>::type batchFilterGenerator,
    const RevisionIds& batchData
) const;

template Revisions ReaderImpl::loadRevisions<std::vector<RelationData>>(
    LoadLimitations loadLimitations,
    const OptionalSnapshotId& snapshotId,
    const OptionalSize& limit,
    const filters::FilterExpr* basicFilter,
    CheckResultSize checkResultSize,
    typename BatchFilterGenerator<std::vector<RelationData>>::type batchFilterGenerator,
    const std::vector<RelationData>& batchData
) const;


//filter-aware version of loading revisionIds
RevisionIds ReaderImpl::loadRevisionIds(
    LoadLimitations loadLimitations,
    const OptionalSnapshotId& snapshotId,
    const OptionalSize& limit,
    const filters::FilterExpr& expr
) const
{
    auto strQuery = db::StructuredQueryProvider::revisionId(
        txn_.conn(),
        expr);
    if (strQuery.subqueries.empty()) {
        return {};
    }

    auto query = makeQuery(strQuery, loadLimitations, snapshotId, limit);

    const auto& pqxxResult = txn_.exec(query);
    auto revisionIds = loadRevisionIdsFromPqxx(pqxxResult);

    std::sort(revisionIds.begin(), revisionIds.end());
    if (needExcludeHistoricals(loadLimitations)) {
        //performing filtering via additional request
        return removeNonHeadRevisions(std::move(revisionIds), snapshotId);
    }

    return revisionIds;
}

//pipelined version of loading revisionIds
RevisionIds ReaderImpl::loadRevisionIds(
    LoadLimitations loadLimitations,
    const OptionalSnapshotId& snapshotId,
    const OptionalSize& limit,
    const DBIDSet& objectIds
) const
{
    using Batch = maps::common::Batch<DBIDSet>;
    auto queryGenerator = [&] (const Batch& batch) -> OptionalString
    {
        auto batchFilter = filters::ObjRevAttr::objectId().in(batch);

        auto query = db::StructuredQueryProvider::revisionId(
            txn_.conn(),
            batchFilter);
        if (query.subqueries.empty()) {
            return std::nullopt;
        }

        return makeQuery(query, loadLimitations, snapshotId, limit);
    };

    RevisionIds revisionIds;
    revisionIds.reserve(objectIds.size());
    auto resultInterpreter = [&](
        const pqxx::result& pqxxResult,
        const Batch& //batch
    )
    {
        RevisionIds batchRevisionIds = loadRevisionIdsFromPqxx(pqxxResult);
        if (revisionIds.empty()) {
            revisionIds.swap(batchRevisionIds);
        } else {
            revisionIds.insert(revisionIds.end(), batchRevisionIds.begin(), batchRevisionIds.end());
        }
    };
    batchLoadData<DBIDSet>(
        txn_,
        objectIds,
        LOAD_REVISION_IDS_BATCH_SIZE,
        queryGenerator,
        resultInterpreter
    );

    revisionIds.shrink_to_fit();
    std::sort(revisionIds.begin(), revisionIds.end());
    if (needExcludeHistoricals(loadLimitations)) {
        //performing in-memory filtering
        return excludeNonHeadRevisions(std::move(revisionIds));
    }

    return revisionIds;
}

//pipelined version of loading revisionIds
RevisionIds ReaderImpl::loadRevisionIds(
    LoadLimitations loadLimitations,
    const OptionalSnapshotId& snapshotId,
    const OptionalSize& limit,
    const DBIDSet& objectIds,
    const filters::FilterExpr& expr
) const
{
    using Batch = maps::common::Batch<DBIDSet>;

    auto queryGenerator = [&] (const Batch& batch) -> OptionalString {
        auto batchFilter = filters::makeFilterForObjectIdBatch(batch) &&
            filters::ProxyReferenceFilterExpr(expr);

        auto query = db::StructuredQueryProvider::revisionId(
            txn_.conn(),
            batchFilter);
        if (query.subqueries.empty()) {
            return std::nullopt;
        }

        return makeQuery(query, loadLimitations, snapshotId, limit);
    };

    RevisionIds revisionIds;
    revisionIds.reserve(objectIds.size());

    auto resultInterpreter = [&](const pqxx::result& pqxxResult, const Batch&) {
        RevisionIds batchRevisionIds = loadRevisionIdsFromPqxx(pqxxResult);
        if (revisionIds.empty()) {
            revisionIds = std::move(batchRevisionIds);
        } else {
            revisionIds.insert(
                revisionIds.end(),
                batchRevisionIds.begin(), batchRevisionIds.end()
            );
        }
    };

    batchLoadData<DBIDSet>(
        txn_,
        objectIds,
        LOAD_REVISION_IDS_BATCH_SIZE,
        queryGenerator,
        resultInterpreter
    );

    revisionIds.shrink_to_fit();
    std::sort(revisionIds.begin(), revisionIds.end());
    if (needExcludeHistoricals(loadLimitations)) {
        //performing filtering via additional request
        return removeNonHeadRevisions(std::move(revisionIds), snapshotId);
    }

    return revisionIds;
}

RevisionIds ReaderImpl::removeNonHeadRevisions(
    RevisionIds&& revisionIds,
    const OptionalSnapshotId& snapshotId
) const
{
    if (revisionIds.empty()) {
        return revisionIds;
    }

    DBIDSet objectIds;
    for (const auto& revisionId: revisionIds) {
        objectIds.insert(revisionId.objectId());
    }

    RevisionIds headRevisionIds = loadRevisionIds(
        LoadLimitations::Snapshot,
        snapshotId,
        NO_LIMIT,
        objectIds
    );

    RevisionIds result;
    std::set_intersection(
        revisionIds.begin(), revisionIds.end(),
        headRevisionIds.begin(), headRevisionIds.end(),
        std::back_inserter(result)
    );
    return result;
}


Revisions ReaderImpl::removeNonHeadRevisions(
    Revisions&& revisions,
    const OptionalSnapshotId& snapshotId
) const
{
    DBIDSet objectIds;
    for (const auto& revision: revisions) {
        objectIds.insert(revision.id().objectId());
    }

    RevisionIds headRevisionIds = loadRevisionIds(
        LoadLimitations::Snapshot,
        snapshotId,
        NO_LIMIT,
        objectIds
    );

    Revisions result;
    auto inserter = std::inserter(result, result.end());
    //g++ implementation of set_intersection
    //doesn't support containers of different type
    //using own impl with the same logic
    //http://en.cppreference.com/w/cpp/algorithm/set_intersection
    auto first1 = revisions.cbegin();
    auto last1 = revisions.cend();
    auto first2 = headRevisionIds.cbegin();
    auto last2 = headRevisionIds.cend();
    while ((first1 != last1) && (first2 != last2)) {
        if (first1->id() < *first2) {
            ++first1;
        } else if (first1->id() > *first2) {
            ++first2;
        } else {
            *inserter = *first1;
            ++first1;
            ++first2;
        }
    }
    return result;
}

} // namespace maps::wiki::revision

