#include "result_interpreter.h"
#include "sql_strings.h"
#include "helpers.h"

#include <algorithm>
#include <cstddef>
#include <unordered_map>

namespace maps::wiki::revision {

namespace {

class NonHeadsExcluder
{
public:
    NonHeadsExcluder(const RevisionIds& revisionIds)
    {
        for (const auto& revisionId: revisionIds) {
            insertRevisionId(revisionId);
        }
    }

    NonHeadsExcluder(const Revisions& revisions)
    {
        for (const auto& revision: revisions) {
            insertRevisionId(revision.id());
        }
    }

    //returns true if revisionId isn't HEAD
    //Should be combined with std::remove_if
    bool operator() (const RevisionID& revisionId) const
    {
        auto it = objectCommitIds_.find(revisionId.objectId());
        return (
            (it == objectCommitIds_.end()) ||
            (it->second != revisionId.commitId())
        );
    }

    //returns true if revision isn't HEAD
    //Should be combined with std::remove_if
    bool operator() (const ObjectRevision& revision) const
    {
        return operator()(revision.id());
    }

private:
    void insertRevisionId(const RevisionID& revisionId)
    {
        auto insertResult = objectCommitIds_.insert({
            revisionId.objectId(),
            revisionId.commitId()
        });

        if (!insertResult.second) {
            insertResult.first->second = std::max(
                insertResult.first->second,
                revisionId.commitId()
            );
        }
    }

    std::unordered_map<DBID, DBID> objectCommitIds_;
};

bool isColumnFetched(const pqxx::result& pqxxResult, const std::string& column)
{
    for (size_t i = 0; i < pqxxResult.columns(); ++i) {
        if (pqxxResult.column_name(i) == column) {
            return true;
        }
    }
    return false;
}

//load helpers functions
std::optional<Attributes> loadAttributes(
    const pqxx::row& tuple
)
{
    const auto& field = tuple[sql::col::ATTRIBUTES];
    if (field.is_null()) {
        return std::nullopt;
    }

    return helpers::stringToAttributes(field.as<std::string>());
}

} //anonymous namespace

RevisionData
loadRevisionDataFromPqxx(const pqxx::row& row)
{
    RevisionData result;
    result.deleted = row[sql::col::IS_DELETED].as<bool>();
    result.attributesId = row[sql::col::ATTRIBUTES_ID].as<DBID>();
    result.descriptionId = row[sql::col::DESCRIPTION_ID].as<DBID>();
    result.geometryId = row[sql::col::GEOMETRY_ID].as<DBID>();

    result.prevCommitId = row[sql::col::PREV_COMMIT_ID].as<DBID>();
    result.nextCommitId = row[sql::col::NEXT_COMMIT_ID].as<DBID>();

    result.masterObjectId = row[sql::col::MASTER_OBJECT_ID].as<DBID>();
    result.slaveObjectId = row[sql::col::SLAVE_OBJECT_ID].as<DBID>();
    return result;
}

RevisionID loadRevisionIdFromPqxx(const pqxx::row& pqxxTuple)
{
    return RevisionID{
        pqxxTuple[sql::col::OBJECT_ID].as<DBID>(),
        pqxxTuple[sql::col::COMMIT_ID].as<DBID>()
    };
}


ObjectData loadObjectDataFromPqxx(
    const pqxx::row& pqxxTuple,
    const RevisionData& revisionData,
    bool attrFetched
)
{
    return ObjectData{
        (attrFetched && revisionData.attributesId) ?
            loadAttributes(pqxxTuple) :
            std::nullopt,
            std::nullopt,
            std::nullopt,
        (revisionData.masterObjectId > 0) ?
            std::make_optional(RelationData{
                revisionData.masterObjectId,
                revisionData.slaveObjectId
            }) :
            std::nullopt,
        revisionData.deleted
    };
}


RevisionIds loadRevisionIdsFromPqxx(const pqxx::result& pqxxResult)
{
    if (pqxxResult.empty()) {
        return {};
    }

    RevisionIds result;
    result.reserve(pqxxResult.size());
    for (const auto& pqxxTuple: pqxxResult) {
        result.emplace_back(loadRevisionIdFromPqxx(pqxxTuple));
    }

    return result;
}


Revisions loadRevisionsFromPqxx(const pqxx::result& pqxxResult)
{
    auto attrFetched = isColumnFetched(pqxxResult, sql::col::ATTRIBUTES);

    Revisions result;
    for (const auto& pqxxTuple: pqxxResult) {
        auto revisionId = loadRevisionIdFromPqxx(pqxxTuple);
        auto revisionData = loadRevisionDataFromPqxx(pqxxTuple);

        result.emplace_back(
            revisionId,
            revisionData,
            loadObjectDataFromPqxx(
                pqxxTuple,
                revisionData,
                attrFetched
            )
        );
    }

    return result;
}


RevisionIdToRevisionMap toRevisionsMap(const Revisions& revisions)
{
    RevisionIdToRevisionMap result;
    for (const auto& revision: revisions) {
        result.insert({revision.id(), revision});
    }
    return result;
}


ObjectIdToRevisionMap toObjectsMap(const Revisions& revisions)
{
    ObjectIdToRevisionMap result;
    for (const auto& revision: revisions) {
        result.insert({revision.id().objectId(), revision});
    }
    return result;
}


RevisionIds excludeNonHeadRevisions(RevisionIds&& data)
{
    NonHeadsExcluder excluder(data);
    data.erase(
        std::remove_if(
            data.begin(),
            data.end(),
            excluder
        ),
        data.end()
    );
    return data;
}

Revisions excludeNonHeadRevisions(Revisions&& data)
{
    NonHeadsExcluder excluder(data);
    data.erase(
        std::remove_if(
            data.begin(),
            data.end(),
            excluder
        ),
        data.end()
    );
    return data;
}

} // namespace maps::wiki::revision

