#include <yandex/maps/wiki/revision/revisionsgateway.h>
#include "revisions_gateway_impl.h"
#include "snapshots_impl.h"
#include "commit_data.h"
#include "commit_worker.h"
#include "conflicts_checker.h"
#include "helpers.h"
#include "objectid_generator.h"
#include "snapshot_id_impl.h"

#include <yandex/maps/wiki/revision/branch.h>
#include <yandex/maps/wiki/revision/branch_manager.h>
#include <yandex/maps/wiki/revision/exception.h>

namespace maps::wiki::revision {

using namespace helpers;

namespace {
const size_t ACQUIRE_IDS_OPTIMIZE_LIMIT = 1000000;
} // namespace

class RevisionsGateway::Impl : public RevisionsGatewayImpl
{
public:
    Impl(Transaction& work, BranchType type, DBID branchId)
        : RevisionsGatewayImpl(work, type, branchId)
    {}

    ObjectIdGenerator objectIdGenerator;
};


RevisionsGateway::RevisionsGateway(Transaction& work)
    : impl_(new Impl(work, BranchType::Trunk, TRUNK_BRANCH_ID))
{}

RevisionsGateway::RevisionsGateway(Transaction& work, const Branch& branch)
    : impl_(new Impl(work, branch.type(), branch.id()))
{}

RevisionsGateway::RevisionsGateway(RevisionsGateway&& other) noexcept
    : impl_(std::move(other.impl_))
{}

RevisionsGateway& RevisionsGateway::operator=(RevisionsGateway&& other) noexcept
{
    if (&other != this) {
        impl_.swap(other.impl_);
    }
    return *this;
}

RevisionsGateway::~RevisionsGateway() = default;

Transaction&
RevisionsGateway::work() const
{
    return impl_->work();
}

DBID
RevisionsGateway::branchId() const
{
    return impl_->branchId();
}

BranchType
RevisionsGateway::branchType() const
{
    return impl_->branchType();
}

RevisionID
RevisionsGateway::acquireObjectId() const
{
    DBID id = impl_->objectIdGenerator.newObjectId(impl_->work());
    return RevisionID::createNewID(id);
}

std::list<ObjectIdRange>
RevisionsGateway::acquireObjectIds(size_t number) const
{
    if (!number) {
        return std::list<ObjectIdRange>();
    }

    //When database is empty and amount of requested objects is huge,
    //an optimization may be applied
    //See newObjectIds implementation for details
    bool optimize = (
        (number > ACQUIRE_IDS_OPTIMIZE_LIMIT) &&
        (impl_->headCommitId() == 0)
    );
    return impl_->objectIdGenerator.newObjectIds(impl_->work(), number, optimize);
}

DBID RevisionsGateway::lastObjectId() const
{
    return impl_->objectIdGenerator.lastObjectId(impl_->work());
}

DBID RevisionsGateway::reserveObjectIds(size_t number) const
{
    return impl_->objectIdGenerator.reserveObjectIds(impl_->work(), number);
}

DBID
RevisionsGateway::headCommitId() const
{
    return impl_->headCommitId();
}

SnapshotId
RevisionsGateway::maxSnapshotId() const
{
    return impl_->maxSnapshotId();
}

Commit
RevisionsGateway::createCommit(
    const ConstRange<NewRevisionData>& newData,
    UserID createdBy,
    const Attributes& attributes) const
{
    REQUIRE(!newData.empty(), "empty commit data");
    checkUserId(createdBy);

    impl_->checkReadOnlyBranch();

    auto& work = impl_->work();
    PreparedCommitAttributes attrs(work, attributes);

    const Branch branch = BranchManager(work).load(branchId());

    if (!branch.isWritingAllowed()) {
        std::stringstream errorMessgage;
        errorMessgage << "commits into "
                      << branch.type() << " branch are forbidden"
                      << ", id: " << branch.id()
                      << ", state: " << branch.state();

        if (branch.state() == BranchState::Progress) {
            throw BranchInProgressException() << errorMessgage.str();
        }
        throw BranchUnavailableException() << errorMessgage.str();
    }
    CommitWorker worker(work, newData, branch.type(), branch.id());
    return Commit(worker.createCommit(createdBy, attrs));
}

void
RevisionsGateway::checkConflicts(const ConstRange<RevisionID>& ids) const
{
    if (ids.empty()) {
        return;
    }

    impl_->checkReadOnlyBranch();
    checkRevisionIds(ids);

    ConflictsChecker checker(impl_->work());
    checker.checkConflicts(branchType(), branchId(),
        QueryGenerator::buildCommitIdToObjectIds(ids).first);
}

void
RevisionsGateway::truncateAll() const
{
    impl_->truncateAll();
    impl_->objectIdGenerator.clear();
}

void
RevisionsGateway::createDefaultBranches() const
{
    impl_->createDefaultBranches();
}

Reader
RevisionsGateway::reader() const
{
    return Reader(impl_->readerImpl());
}

Snapshot
RevisionsGateway::snapshot(DBID commitId) const
{
    return Snapshot(std::make_shared<SnapshotImpl>(
        impl_->readerImpl(),
        PImplFactory::create<SnapshotId>(
            std::nullopt,
            commitId
        )
    ));
}

Snapshot
RevisionsGateway::stableSnapshot(DBID commitId) const
{
    return Snapshot(std::make_shared<SnapshotImpl>(
        impl_->readerImpl(),
        SnapshotId::fromCommit(
            commitId,
            impl_->readerImpl()->branchType(),
            work()
        )
    ));
}

Snapshot
RevisionsGateway::snapshot(const SnapshotId& id) const
{
    return Snapshot(std::make_shared<SnapshotImpl>(
        impl_->readerImpl(),
        id
    ));
}

HistoricalSnapshot
RevisionsGateway::historicalSnapshot(DBID maxCommitId) const
{
    return HistoricalSnapshot(std::make_shared<HistoricalSnapshotImpl>(
        impl_->readerImpl(),
        0,
        maxCommitId
    ));
}

HistoricalSnapshot
RevisionsGateway::historicalSnapshot(DBID minCommitId, DBID maxCommitId) const
{
    return HistoricalSnapshot(std::make_shared<HistoricalSnapshotImpl>(
        impl_->readerImpl(),
        minCommitId,
        maxCommitId
    ));
}

} // namespace maps::wiki::revision
