#include <yandex/maps/wiki/revision/snapshot.h>
#include <yandex/maps/wiki/revision/range_helpers.h>
#include "snapshots_impl.h"
#include "filter_generators.h"
#include "helpers.h"
#include <maps/libs/common/include/exception.h>

#include <utility>

namespace maps::wiki::revision {

using namespace helpers;

Snapshot::Snapshot(std::shared_ptr<SnapshotImpl> impl)
    : impl_(std::move(impl))
{}

Snapshot::~Snapshot() = default;

Reader
Snapshot::reader() const
{
    return Reader(impl_->reader);
}

DBID
Snapshot::maxCommitId() const
{
    return impl_->id.commitId();
}

RevisionIds
Snapshot::revisionIdsByRegion(double x1, double y1, double x2, double y2) const
{
    if (impl_->id.empty()) {
        return {};
    }

    return impl_->reader->loadRevisionIds(
        LoadLimitations::Snapshot,
        impl_->id,
        NO_LIMIT,
        filters::GeomFilterExpr(
            filters::GeomFilterExpr::Operation::Intersects,
            x1, y1, x2, y2
        )
    );
}

Revisions
Snapshot::loadMasterRelations(DBID objectId) const
{
    checkObjectId(objectId);
    if (impl_->id.empty()) {
        return {};
    }
    return loadMasterRelations(DBIDSet{objectId});
}

Revisions
Snapshot::loadMasterRelations(const ConstRange<DBID>& objectIds) const
{
    return loadMasterRelations(
        objectIds, filters::ObjRevAttr::isNotDeleted());
}

Revisions
Snapshot::loadMasterRelations(
    const ConstRange<DBID>& objectIds,
    const filters::FilterExpr& expr) const
{
    if (impl_->id.empty() || objectIds.empty()) {
        return {};
    }

    using Batch = maps::common::Batch<DBIDSet>;

    return impl_->reader->loadRevisions(
        LoadLimitations::Snapshot,
        impl_->id,
        NO_LIMIT,
        &expr,
        CheckResultSize::No,
        filters::makeFilterForSlaveObjectIdBatch<Batch>,
        idsToSet(objectIds)
    );
}

Revisions
Snapshot::loadSlaveRelations(DBID objectId) const
{
    checkObjectId(objectId);
    if (impl_->id.empty()) {
        return {};
    }
    return loadSlaveRelations(DBIDSet{objectId});
}

Revisions
Snapshot::loadSlaveRelations(const ConstRange<DBID>& objectIds) const
{
    return loadSlaveRelations(
        objectIds, filters::ObjRevAttr::isNotDeleted());
}

Revisions
Snapshot::loadSlaveRelations(
    const ConstRange<DBID>& objectIds,
    const filters::FilterExpr& expr) const
{
    if (impl_->id.empty() || objectIds.empty()) {
        return {};
    }

    using Batch = maps::common::Batch<DBIDSet>;

    return impl_->reader->loadRevisions(
        LoadLimitations::Snapshot,
        impl_->id,
        NO_LIMIT,
        &expr,
        CheckResultSize::No,
        filters::makeFilterForMasterObjectIdBatch<Batch>,
        idsToSet(objectIds)
    );
}

std::map<DBID, ObjectRevision>
Snapshot::objectRevisions(const ConstRange<DBID>& objectIds) const
{
    if (impl_->id.empty()) {
        return {};
    }

    auto revisions = impl_->reader->loadRevisions(
        LoadLimitations::Snapshot,
        impl_->id,
        NO_LIMIT,
        idsToSet(objectIds)
    );

    std::map<DBID, ObjectRevision> result;
    for (auto&& revision: revisions) {
        DBID objectId = revision.id().objectId();
        REQUIRE(
            result.emplace(objectId, std::move(revision)).second,
            "While loading objectRevisions: received duplicated "
                "result object id " << objectId
        );
    }
    return result;
}

RevisionIds
Snapshot::objectRevisionIds(const ConstRange<DBID>& objectIds) const
{
    if (impl_->id.empty()) {
        return {};
    }
    return impl_->reader->loadRevisionIds(
        LoadLimitations::Snapshot,
        impl_->id,
        NO_LIMIT,
        idsToSet(objectIds)
    );
}

std::optional<ObjectRevision>
Snapshot::objectRevision(DBID objectId) const
{
    checkObjectId(objectId);
    if (impl_->id.empty()) {
        return std::nullopt;
    }

    auto revisions = impl_->reader->loadRevisions(
        LoadLimitations::Snapshot,
        impl_->id,
        NO_LIMIT,
        filters::ObjRevAttr::objectId() == objectId
    );

    if (revisions.empty()) {
        return std::nullopt;
    }
    return revisions.front();
}

RevisionIds
Snapshot::revisionIdsByFilter(const filters::FilterExpr& expr) const
{
    if (impl_->id.empty()) {
        return {};
    }
    return impl_->reader->loadRevisionIds(
        LoadLimitations::Snapshot,
        impl_->id,
        NO_LIMIT,
        expr
    );
}

RevisionIds
Snapshot::revisionIdsByFilter(const ConstRange<DBID>& objectIds,
    const filters::FilterExpr& expr) const
{
    if (impl_->id.empty()) {
        return {};
    }
    return impl_->reader->loadRevisionIds(
        LoadLimitations::Snapshot,
        impl_->id,
        NO_LIMIT,
        idsToSet(objectIds),
        expr
    );
}

std::optional<RevisionIds>
Snapshot::tryLoadRevisionIdsByFilter(
    const filters::FilterExpr& expr, size_t limit) const
{
    checkLimit(limit);
    if (impl_->id.empty()) {
        return RevisionIds();
    }
    auto result = impl_->reader->loadRevisionIds(
        LoadLimitations::Snapshot,
        impl_->id,
        limit + 1,
        expr
    );
    if (result.size() > limit) {
        return std::nullopt;
    }
    return result;
}

Revisions
Snapshot::relationsByFilter(const filters::FilterExpr& expr) const
{
    if (impl_->id.empty()) {
        return {};
    }

    return impl_->reader->loadRevisions(
        LoadLimitations::Snapshot,
        impl_->id,
        NO_LIMIT,
        (
            filters::ProxyReferenceFilterExpr(expr) &&
            filters::ObjRevAttr::isRelation()
        )
    );
}

std::optional<Revisions>
Snapshot::tryLoadRelationsByFilter(
    const filters::FilterExpr& expr, size_t limit) const
{
    checkLimit(limit);
    if (impl_->id.empty()) {
        return Revisions();
    }

    auto result = impl_->reader->loadRevisions(
        LoadLimitations::Snapshot,
        impl_->id,
        limit + 1,
        (
            filters::ProxyReferenceFilterExpr(expr) &&
            filters::ObjRevAttr::isRelation()
        )
    );

    if (result.size() > limit) {
        return std::nullopt;
    }
    return result;
}

Revisions
Snapshot::objectRevisionsByFilter(const filters::FilterExpr& expr) const
{
    if (impl_->id.empty()) {
        return {};
    }

    return impl_->reader->loadRevisions(
        LoadLimitations::Snapshot,
        impl_->id,
        NO_LIMIT,
        expr
    );
}

} // namespace maps::wiki::revision
