#include <maps/wikimap/mapspro/libs/revision_meta/include/approved_queue.h>

#include "preapproved_commits_relations.h"

#include <yandex/maps/wiki/common/pg_utils.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/revision/commit_manager.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/common/include/make_batches.h>

#include <maps/libs/enum_io/include/enum_io.h>

using namespace std::string_literals;

namespace maps::wiki::revision_meta {

namespace {

const size_t DELETE_BATCH_SIZE = 1000;

namespace table_name {
const auto PREAPPROVE = "revision_meta.preapproved_commits_queue"s;
const auto COMMITS = "revision_meta.approved_commits_queue"s;
const auto LABELS = "revision_meta.approved_labels_queue"s;
const auto BBOXES = "revision_meta.approved_bboxes_queue"s;
} // namespace table_name

namespace col {
const auto COMMIT_ID = "commit_id"s;
} // namespace col

const std::string&
modeToTableName(ApprovedQueueMode mode)
{
    switch (mode) {
    case ApprovedQueueMode::PreApprove:
        return table_name::PREAPPROVE;
    case ApprovedQueueMode::ViewAttrs:
        return table_name::COMMITS;
    case ApprovedQueueMode::Labels:
        return table_name::LABELS;
    case ApprovedQueueMode::Bboxes:
        return table_name::BBOXES;
    }
    throw LogicError() << "Unreachable code"s;
}

TCommitIds
loadCommitIds(
    pqxx::transaction_base& txn,
    ApprovedQueueMode mode,
    const std::string& whereClause)
{
    auto query =
        "SELECT "s + col::COMMIT_ID + " FROM "s + modeToTableName(mode);
    if (!whereClause.empty()) {
        query += " WHERE "s + whereClause;
    }

    TCommitIds commitIds;
    for (const auto& row : txn.exec(query)) {
        commitIds.insert(row[0].as<TCommitId>());
    }
    return commitIds;
}

} // namespace

constexpr enum_io::Representations<ApprovedQueueMode> APPROVED_QUEUE_MODE_REPRESENTATION {
    {ApprovedQueueMode::PreApprove, "PreApprove"},
    {ApprovedQueueMode::ViewAttrs, "ViewAttrs"},
    {ApprovedQueueMode::Labels, "Labels"},
    {ApprovedQueueMode::Bboxes, "Bboxes"},
};
DEFINE_ENUM_IO(ApprovedQueueMode, APPROVED_QUEUE_MODE_REPRESENTATION);

TCommitIds
ApprovedQueue::readCommitIds(const TCommitIds& commitIds)
{
    if (commitIds.empty()) {
        return {};
    }
    return loadCommitIds(txn_, mode_, common::whereClause(col::COMMIT_ID, commitIds));
}

TCommitIds
ApprovedQueue::readAllCommitIds()
{
    return loadCommitIds(txn_, mode_, {});
}

size_t
ApprovedQueue::size()
{
    auto result = txn_.exec("SELECT COUNT(*) FROM " + modeToTableName(mode_));
    return result[0][0].as<size_t>();
}

void
ApprovedQueue::deleteCommits(const TCommitIds& commitIds)
{
    for (const auto& batch : maps::common::makeBatches(commitIds, DELETE_BATCH_SIZE)) {
        TCommitIds batchCommitIds(batch.begin(), batch.end());
        auto query =
            "DELETE FROM "s + modeToTableName(mode_) +
            " WHERE "s + common::whereClause(col::COMMIT_ID, batchCommitIds);
        txn_.exec(query);
    }
}

void
ApprovedQueue::push(const TCommitIds& commitIds)
{
    if (commitIds.empty()) {
        return;
    }

    auto query =
        "INSERT INTO "s + modeToTableName(mode_) +
        " VALUES ("s + common::join(commitIds, "),("s) + ")"s;
    txn_.exec(query);
}

PreApprovedQueue::PreApprovedQueue(pqxx::transaction_base& txn)
    : ApprovedQueue(txn, ApprovedQueueMode::PreApprove)
{}

void
PreApprovedQueue::push(const TCommitIds& commitIds)
{
    ApprovedQueue::push(commitIds);

    for (const auto commitId: commitIds) {
        auto relatesTo =
            revision::CommitManager(txn_).findAllDraftContributingCommits({commitId});
        relatesTo.erase(commitId);
        addCommitRelations(
            txn_,
            commitId,
            relatesTo.cbegin(),
            relatesTo.cend()
        );
    }
}

void
PreApprovedQueue::pushServiceTaskCommits(const TCommitIds& commitIds)
{
    push(commitIds);

    addCommitRelations(
        txn_,
        *commitIds.cbegin(),
        std::next(commitIds.cbegin()),
        commitIds.cend()
    );
}

void
PreApprovedQueue::deleteCommits(const TCommitIds& commitIds)
{
    ApprovedQueue::deleteCommits(commitIds);

    for (const auto& batch: maps::common::makeBatches(commitIds, DELETE_BATCH_SIZE)) {
        deleteCommitRelations(txn_, TCommitIds(batch.begin(), batch.end()));
    }
}

std::chrono::seconds
PreApprovedQueue::oldestCommitAge()
{
    auto result = txn_.exec(
R"(SELECT
  COALESCE(
    EXTRACT(EPOCH FROM (NOW() - MIN(added_at)))::integer,
    0)
FROM )" + table_name::PREAPPROVE);

    return std::chrono::seconds(result[0][0].as<std::chrono::seconds::rep>());
}

} // namespace maps::wiki::revision_meta
