#pragma once

#include <set>
#include <stack>
#include <vector>
#include <unordered_map>
#include <unordered_set>

namespace maps::wiki::revision_meta {

template<typename Value>
struct Graph {
public:
    using Component = std::set<Value>;
    using Components = std::vector<Component>;

    void addVertex(Value value) {
        addVertexAndGetId(value);
    }

    void addEdge(Value from, Value to) {
        auto fromId = addVertexAndGetId(from);
        auto   toId = addVertexAndGetId(to);

        vertexes[fromId].alignedVertexIds.insert(  toId);
        vertexes[  toId].alignedVertexIds.insert(fromId);
    }

    Components components() const {
        Components result;

        VertexIds processedVertexIds;
        for (size_t vertexIx = 0; vertexIx < vertexes.size(); ++vertexIx) {
            if (processedVertexIds.count(vertexIx)) {
                continue;
            }

            auto component = lookForNewComponentAt(vertexIx, processedVertexIds);
            if (!component.empty()) {
                result.emplace_back(std::move(component));
            }
        }

        return result;
    }

private:
    using VertexIds = std::unordered_set<size_t>;
    struct Vertex {
        Value value;
        VertexIds alignedVertexIds;
    };
    using Vertexes = std::vector<Vertex>;
    using ValueToVertexId = std::unordered_map<Value, size_t>;

    size_t addVertexAndGetId(Value value) {
        const auto valueToVertexIdIt = valueToVertexId.find(value);
        if (valueToVertexIdIt != valueToVertexId.cend()) {
            return valueToVertexIdIt->second;
        }

        vertexes.emplace_back(Vertex{value, VertexIds{}});
        const auto vertexId = vertexes.size() - 1;
        valueToVertexId[value] = vertexId;

        return vertexId;
    }

    Component lookForNewComponentAt(
        size_t vertexIx,
        VertexIds& processedVertexIds) const
    {
        if (processedVertexIds.count(vertexIx)) {
            return {};
        }

        Component result;

        std::stack<size_t> vertexesToBeProcessed;
        vertexesToBeProcessed.push(vertexIx);
        while (!vertexesToBeProcessed.empty()) {
            const auto curVertexIx = vertexesToBeProcessed.top();
            vertexesToBeProcessed.pop();

            if (processedVertexIds.count(curVertexIx)) {
                continue;
            }
            processedVertexIds.insert(curVertexIx);

            const auto& curVertex = vertexes[curVertexIx];
            result.emplace(curVertex.value);

            for (const auto alignedVertexId: curVertex.alignedVertexIds) {
                vertexesToBeProcessed.emplace(alignedVertexId);
            }
        }

        return result;
    }

    Vertexes vertexes;
    ValueToVertexId valueToVertexId;
};

} // namespace maps::wiki::revision_meta
