#include "preapproved_commits_relations.h"

#include <yandex/maps/wiki/common/pg_utils.h>
#include <yandex/maps/wiki/common/string_utils.h>

namespace maps::wiki::revision_meta {

namespace {

namespace table {
const std::string RELATIONS = "revision_meta.preapproved_commits_relations";
} // namespace table

namespace col {
const std::string COMMIT_ID = "commit_id";
const std::string RELATES_TO = "relates_to";
} // namespace col

} // namespace

void addCommitRelations(
    pqxx::transaction_base& txnCore,
    TCommitId commitId,
    TCommitIds::const_iterator relatesToCommitIdsBegin,
    TCommitIds::const_iterator relatesToCommitIdsEnd)
{
    if (relatesToCommitIdsBegin == relatesToCommitIdsEnd) {
        return;
    }

    const auto commitIdStr = std::to_string(commitId);
    txnCore.exec(
        "INSERT INTO " + table::RELATIONS + " (" + col::COMMIT_ID + "," + col::RELATES_TO + ") VALUES " +
        common::join(
            relatesToCommitIdsBegin,
            relatesToCommitIdsEnd,
            [&](auto relatesToCommitId) {
                return "(" + commitIdStr + "," + std::to_string(relatesToCommitId) + ")";
            },
            ","
        ) +
        " ON CONFLICT (" + col::COMMIT_ID + "," + col::RELATES_TO + ") DO NOTHING"
    );
}

void deleteCommitRelations(
    pqxx::transaction_base& txnCore,
    const TCommitIds& commitIds)
{
    if (commitIds.empty()) {
        return;
    }

    txnCore.exec(
        "WITH to_delete AS (\n"
        "  SELECT * FROM " + table::RELATIONS + "\n"
        "  WHERE\n"
        "    " + common::whereClause(col::COMMIT_ID, commitIds)  + " OR\n"
        "    " + common::whereClause(col::RELATES_TO, commitIds) + "\n"
        "  ORDER BY " + col::COMMIT_ID + ", " + col::RELATES_TO + "\n"
        "  FOR UPDATE\n"
        ")\n"
        "DELETE FROM " + table::RELATIONS + "\n"
        "USING to_delete\n"
        "WHERE\n" +
        "  " + table::RELATIONS + "." + col::COMMIT_ID  + " = to_delete." + col::COMMIT_ID + " AND\n"
        "  " + table::RELATIONS + "." + col::RELATES_TO + " = to_delete." + col::RELATES_TO
    );
}

Relations getAllRelations(pqxx::transaction_base& txnCore)
{
    const auto rows = txnCore.exec(
        "SELECT " + col::COMMIT_ID + ", " + col::RELATES_TO + " "
        "FROM " + table::RELATIONS
    );

    Relations result;
    result.reserve(rows.size());
    for (const auto& row: rows) {
        result.emplace(
            Relation{
                row["commit_id"].as<TCommitId>(),
                row["relates_to"].as<TCommitId>()
            }
        );
    }

    return result;
}

} // namespace maps::wiki::revision_meta
