#pragma once

#include <yandex/maps/wiki/revision/branch.h>
#include <yandex/maps/wiki/revision/filters.h>

#include <memory>
#include <string>
#include <vector>
#include <set>
#include <iosfwd>

namespace maps::wiki::revisionapi {

enum class RelationsExportFlags
{
    None = 0,
    MasterToSlave = (1 << 0),
    SlaveToMaster = (1 << 1),
    SkipDangling = (1 << 2)
};

std::ostream& operator << (std::ostream& os, RelationsExportFlags flags);

constexpr RelationsExportFlags operator&(
        RelationsExportFlags lhs,
        RelationsExportFlags rhs)
{
    return static_cast<RelationsExportFlags>(
        static_cast<int>(lhs) & static_cast<int>(rhs));
}

constexpr RelationsExportFlags operator|(
        RelationsExportFlags lhs,
        RelationsExportFlags rhs)
{
    return static_cast<RelationsExportFlags>(
        static_cast<int>(lhs) | static_cast<int>(rhs));
}

constexpr bool operator!(RelationsExportFlags f)
{ return f == RelationsExportFlags::None; }

enum class EmptyJsonPolicy { Export = 0, Skip = 1 };

using FilterPtr = std::shared_ptr<revision::filters::FilterExpr>;

using Strings = std::vector<std::string>;

class ExportParams
{
public:
    /**
     * Constructs export parameters object.
     * By default, no filtering set.
     *
     * @param branch - branch to export objects from
     * @param commitId - commit ID
     * @param flags - set of zero or more flags defining how relations
     *                will be exported
     * @param invertDirectionRoles - special relations
     */
    ExportParams(
        const revision::Branch& branch,
        revision::DBID commitId,
        RelationsExportFlags flags,
        const Strings& invertDirectionRoles = Strings{});

    /**
     * Constructs export parameters object.
     *
     * @param branch - branch to export objects from
     * @param commitId - commit ID
     * @param jsonConfigPath -
     *     file content example:
     *     @code{.json}
     *     {
     *         "attributes": {
     *             "relations_export_mode": "Master <== Slave",
     *             "invert_direction_for_roles": "end,from,start,to,via"
     *         },
     *         "categories": [
     *             {
     *                 "category": "rd_jc"
     *             },
     *         ],
     *         ...
     *         "relations": [
     *             {
     *                 "master": "hydro_fc_el",
     *                 "role": "end",
     *                 "slave": "hydro_fc_jc",
     *                 "direction": "Master ==> Slave"
     *             },
     *         ]
     *         ...
     *     }
     *     @endcode
     */
    static
    std::unique_ptr<ExportParams> loadFromFile(
        const revision::Branch& branch,
        revision::DBID commitId,
        const std::string& jsonConfigPath);

    /**
     * Constructs export parameters object.
     *
     * @param branch - branch to export objects from
     * @param commitId - commit ID
     * @param categoriesForFilter - categories to export
     * @param allowedRelativesCategories - Additional categories of relatives to export relations with
     */
    ExportParams(
        const revision::Branch& branch,
        revision::DBID commitId,
        const Strings& categoriesForFilter,
        const Strings& allowedRelativesCategories);

    ~ExportParams();

    // sets number of objects written to a single output stream
    // defaults to 0 = write all objects to 0-th stream
    void setWriteBatchSize(size_t size);

    void setEmptyJsonPolicy(EmptyJsonPolicy policy);

    const revision::Branch& branch() const;
    revision::DBID commitId() const;
    RelationsExportFlags flags() const;
    size_t writeBatchSize() const;

    const FilterPtr& filter() const;

    const std::set<std::string>& categories() const;
    const std::set<std::string>& categoriesForFilter() const;
    const std::set<std::string>& allowedRelativesCategories() const;

    const Strings& invertDirectionRoles() const;
    EmptyJsonPolicy emptyJsonPolicy() const;

private:
    class Impl;
    std::unique_ptr<Impl> impl_;
};

// Helper class for single-chunk output
class SingleStreamWrapper
{
    template<typename T>
    static void nopDeleter (T*) {}

public:
    using return_type = std::shared_ptr<std::ostream>;

    SingleStreamWrapper(std::ostream& os)
        : ret_(&os, &nopDeleter<std::ostream>)
    { }

    return_type operator ()(size_t) const
    { return ret_; }

private:
    return_type ret_;
};

} // namespace maps::wiki::revisionapi
