#pragma once

#include <yandex/maps/wiki/revisionapi/export_params.h>

#include <maps/libs/pgpool/include/pgpool3.h>
#include <maps/libs/common/include/exception.h>
#include <yandex/maps/wiki/revision/branch.h>
#include <yandex/maps/wiki/revision/filters.h>
#include <yandex/maps/wiki/revision/revisionid.h>

#include <functional>
#include <iostream>
#include <memory>
#include <vector>

namespace maps::wiki::revisionapi {

enum class VerboseLevel
{
    Brief,
    Full
};

enum class IdMode
{
    PreserveId,
    IgnoreJsonId,
    StartFromJsonId
};

class BaseIdMapper {
public:
    virtual ~BaseIdMapper() = default;
    virtual revision::RevisionID getRevisionId(const std::string& jsonId) const = 0;
    virtual revision::RevisionID acquireNewRevisionId() const = 0;
};

using GetStreamForChunkFunc = std::function<std::shared_ptr<std::ostream> (size_t)>;

class DataError: public Exception {
    using Exception::Exception;
};

class RevisionAPI {
public:
    /**
     * @param pool - pgpool3
     * @param verboseLevel - dump errors into stderr on process data
     */
    explicit RevisionAPI(
        pgpool3::Pool& pool,
        VerboseLevel verboseLevel = VerboseLevel::Full);

    ~RevisionAPI();

    /**
     * Creates new commit in the specified branch with data from istream.
     *
     * @param userId - user ID
     *
     * @param idMode - id processing mode:
     *          PreserveId      - use ids from json in result data, sequence for relation Ids
     *                            is supposed to be already set to proper value
     *          IgnoreJsonId    - ignore ids from json, generate new ids for result data
     *          StartFromJsonId - use max object Id from json as start for relation Ids
     *
     * @param is - json-data
     *
     * @param batchLoadingSize - number of objects in a single batch, imported to
     *                           the database. By default loads all objects at
     *                           once, without batching.
     *
     * @param idOffset - offset to be added to all object ids, ignored in all id modes except for
     *                   PreserveId
     *
     * @return list of commit IDs
     */
    std::list<revision::DBID> importData(
        revision::UserID userId,
        IdMode idMode,
        std::istream& is,
        size_t batchLoadingSize = 0,
        size_t idOffset = 0);

    std::list<revision::DBID> importData(
        revision::UserID userId,
        IdMode idMode,
        std::istream& is,
        std::map<std::string, std::string>& jsonIdToError,
        pgpool3::TransactionHandle& writeTr,
        size_t batchLoadingSize = 0,
        size_t idOffset = 0);

    std::list<revision::DBID> importData(
        revision::UserID userId,
        const BaseIdMapper& idMapper,
        std::istream& is,
        std::map<std::string, std::string>& jsonIdToError,
        pgpool3::TransactionHandle& writeTr,
        size_t batchLoadingSize = 0);

    /**
     * Exports objects as json with specified parameters set.
     *
     * @param params - set of export parameters @see ExportParams
     * @param getStreamForChunk - callback which must return output stream for
     *                            writing chunk of objects number n, zero-based
     * @param filter / objectIds
     */
    void exportData(
        const ExportParams& params,
        GetStreamForChunkFunc getStreamForChunkFunc,
        const FilterPtr& filter = FilterPtr{});
    void exportData(
        const ExportParams& params,
        GetStreamForChunkFunc getStreamForChunkFunc,
        const std::vector<revision::DBID>& objectIds);

    /**
     * Creates new commit deleting objects from objectIds.
     *
     * @param userId - user ID
     *
     * @param objectIds - objects to be deleted
     *
     * @param commitBatchSize - number of objects in a single batch. By default
     *                          loads all objects at once, without batching.
     *
     * @return list of commit IDs
     */
    std::list<revision::DBID> deleteData(
        revision::UserID userId,
        const std::vector<revision::DBID>& objectIds,
        size_t commitBatchSize = 0);

    std::list<revision::DBID> deleteDataFromBranch(
        revision::DBID branchId,
        revision::UserID userId,
        const std::vector<revision::DBID>& objectIds,
        size_t commitBatchSize = 0);

private:
    class Impl;
    std::unique_ptr<Impl> impl_;
}; //class RevisionAPI

} // namespace maps::wiki::revisionapi
