#include <yandex/maps/wiki/social/comments_feed.h>

#include "factory.h"
#include "helpers.h"
#include "magic_strings.h"

#include <maps/libs/common/include/exception.h>
#include <yandex/maps/wiki/common/string_utils.h>

#include <array>

namespace maps::wiki::social {

namespace {

std::string
selectClause(const std::string& where)
{
    static const std::array COMMENT_COLUMNS{
        sql::col::ID,
        sql::col::TYPE,
        sql::col::OBJECT_ID,
        sql::col::FEEDBACK_TASK_ID,
        sql::col::COMMIT_ID,
        sql::col::CREATED_BY,
        sql::col::CREATED_AT,
        sql::col::DELETED_BY,
        sql::col::DELETED_AT,
        sql::col::INTERNAL,
        sql::col::DATA
    };

    return "SELECT " + common::join(COMMENT_COLUMNS, ',') +
        " FROM " + sql::table::COMMENT + " " + where;
}

std::string
limitClause(size_t limit)
{
    checkLimit(limit);
    return " LIMIT " + std::to_string(limit);
}

const std::string ORDER_BY_COMMENT_ID_DESC =
    " ORDER BY " + sql::table::COMMENT + "." + sql::col::ID + " DESC ";

} //namespace

CommentsFeedParams&
CommentsFeedParams::setCreatedBy(TUid createdBy)
{
    createdBy_ = createdBy;
    return *this;
}

CommentsFeedParams&
CommentsFeedParams::setObjectId(TId objectId)
{
    objectId_ = objectId;
    return *this;
}

CommentsFeedParams&
CommentsFeedParams::setCommitId(TId commitId)
{
    commitId_ = commitId;
    return *this;
}

CommentsFeedParams&
CommentsFeedParams::setFeedbackTaskId(std::optional<TId> feedbackTaskId)
{
    feedbackTaskId_ = feedbackTaskId;
    return *this;
}

CommentsFeedParams&
CommentsFeedParams::setAoiId(TId aoiId)
{
    aoiId_ = aoiId;
    return *this;
}

CommentsFeedParams&
CommentsFeedParams::setTypes(const CommentTypes& types)
{
    types_ = types;
    return *this;
}

CommentsFeedParams&
CommentsFeedParams::setDeletedPolicy(comments::DeletedPolicy deletedPolicy)
{
    deletedPolicy_ = deletedPolicy;
    return *this;
}

CommentsFeedParams&
CommentsFeedParams::setInternal(std::optional<Comment::Internal> internal)
{
    internal_ = internal;
    return *this;
}

CommentsFeed::CommentsFeed(
    pqxx::transaction_base& work,
    const CommentsFeedParams& params)
    : work_(work)
    , params_(params)
{}

std::string
CommentsFeed::whereClause() const
{
    std::list<std::string> conditions;

    if (params_.createdBy()) {
        conditions.push_back(
            sql::col::CREATED_BY + "=" + std::to_string(params_.createdBy()));
    }
    if (params_.objectId()) {
        conditions.push_back(sql::col::OBJECT_ID + ">0");
        conditions.push_back(
            sql::col::OBJECT_ID + "=" + std::to_string(params_.objectId()));
    }
    if (params_.commitId()) {
        conditions.push_back(
            sql::col::COMMIT_ID + "=" + std::to_string(params_.commitId()));
    }
    if (params_.feedbackTaskId()) {
        conditions.push_back(
            sql::col::FEEDBACK_TASK_ID + "=" + std::to_string(*params_.feedbackTaskId()));
    }
    if (params_.aoiId()) {
        conditions.push_back(
            sql::col::ID + " IN (SELECT " + sql::col::COMMENT_ID + " FROM " +
                sql::table::AOI_COMMENT + " WHERE " + sql::col::AOI_ID + "=" +
                std::to_string(params_.aoiId()) + ")");
    }
    if (!params_.types().empty()) {
        conditions.push_back(
            sql::col::TYPE + " IN ('" + common::join(params_.types(), "','") + "')");
    }

    if (params_.deletedPolicy() == comments::DeletedPolicy::Hide) {
        conditions.push_back(sql::col::DELETED_BY + "=0");
    }

    if (params_.internal()) {
        switch (*params_.internal()) {
            case Comment::Internal::No:
                conditions.push_back(sql::col::INTERNAL + "=" + sql::value::FALSE);
                break;
            case Comment::Internal::Yes:
                conditions.push_back(sql::col::INTERNAL + "=" + sql::value::TRUE);
                break;
        }
    }

    return "WHERE " + common::join(conditions, " AND ");
}

size_t
CommentsFeed::count() const
{
    auto query =
        "SELECT COUNT(*)"
        " FROM " + sql::table::COMMENT + " " + whereClause();
    auto r = work_.exec(query);
    ASSERT(r.size() == 1);
    return (r[0][0]).as<size_t>();
}

std::pair<Comments, HasMore>
CommentsFeed::commentsHead(size_t limit) const
{
    auto query =
        selectClause(whereClause()) +
        ORDER_BY_COMMENT_ID_DESC +
        limitClause(limit + 1);
    auto comments = Factory::comments(work_.exec(query));
    auto hasMore = HasMore::No;
    if (comments.size() > limit) {
        hasMore = HasMore::Yes;
        comments.pop_back();
    }
    return std::make_pair(std::move(comments), hasMore);
}

std::pair<Comments, HasMore>
CommentsFeed::commentsNewer(TId commentId, size_t limit) const
{
    checkLimit(limit);
    ASSERT(commentId);

    auto query =
        selectClause(whereClause()) +
        " AND " + sql::col::ID + " > " + std::to_string(commentId) +
        " ORDER BY " + sql::table::COMMENT + "." + sql::col::ID + " ASC " +
        limitClause(limit + 1);

    auto comments = Factory::comments(work_.exec(query));

    auto hasMore = HasMore::No;
    if (comments.size() > limit) {
        hasMore = HasMore::Yes;
        comments.pop_back();
    }
    std::reverse(comments.begin(), comments.end());
    return std::make_pair(std::move(comments), hasMore);
}

std::pair<Comments, HasMore>
CommentsFeed::commentsOlder(TId commentId, size_t limit) const
{
    checkLimit(limit);
    ASSERT(commentId);
    auto query =
        selectClause(whereClause()) +
        " AND " + sql::col::ID + " < " + std::to_string(commentId) +
        ORDER_BY_COMMENT_ID_DESC +
        limitClause(limit + 1);
    auto comments = Factory::comments(work_.exec(query));
    auto hasMore = HasMore::No;
    if (comments.size() > limit) {
        hasMore = HasMore::Yes;
        comments.pop_back();
    }
    return std::make_pair(std::move(comments), hasMore);
}

} // namespace maps::wiki::social
