#include <yandex/maps/wiki/social/event.h>

#include "helpers.h"
#include "magic_strings.h"

#include <maps/libs/common/include/exception.h>
#include <maps/libs/geolib/include/conversion.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/revision/common.h>

#include <boost/lexical_cast.hpp>

namespace maps::wiki::social {

namespace {

const size_t COORDINATE_PRECISION = 12;
const double MAX_LAT = 89.0;

inline double adjustLatitude(const double& lat)
{
    return std::max(-MAX_LAT, std::min(MAX_LAT, lat));
}

geolib3::BoundingBox
jsonToBbox(const std::string& boundsJson)
{
    REQUIRE(boundsJson.size() >= 9, "invalid bounds json: '" << boundsJson << "'");
    REQUIRE(*boundsJson.begin() == '[' && *boundsJson.rbegin() == ']',
            "invalid bounds json: '" << boundsJson << "'");

    std::vector<double> coords;
    std::string str;
    for (size_t i = 1; i < boundsJson.length(); ++i) {
        char c = boundsJson[i];
        if (c == ',' || c == ']') {
            coords.push_back(std::stod(str));
            str.clear();
        } else {
            str += c;
        }
    }
    REQUIRE(coords.size() == 4, "invalid bounds json: '" << boundsJson << "'");
    geolib3::BoundingBox bbox(
            geoPoint2Mercator(geolib3::Point2(coords[0], adjustLatitude(coords[1]))),
            geoPoint2Mercator(geolib3::Point2(coords[2], adjustLatitude(coords[3]))));
    REQUIRE(std::isfinite(bbox.minX()) && std::isfinite(bbox.minY())
            && std::isfinite(bbox.maxX()) && std::isfinite(bbox.maxY()),
           "invalid bbox for json: '" << boundsJson << "' after transforming to mercator");
    if (bbox.isDegenerate()) {
        bbox = geolib3::resizeByValue(bbox, 1e-3);
    }
    return bbox;
}

void
createEventExtraData(
    pqxx::transaction_base& txn,
    TId eventId,
    const std::optional<EventExtraData>& data)
{
    if (!data) {
        return;
    }

    std::vector<std::string> columns;
    std::vector<std::string> values;

    if (data->ftTypeId) {
        columns.emplace_back(sql::col::FT_TYPE_ID);
        values.emplace_back(std::to_string(*data->ftTypeId));
    }
    if (data->businessRubricId) {
        columns.emplace_back(sql::col::BUSINESS_RUBRIC_ID);
        values.emplace_back(std::to_string(*data->businessRubricId));
    }

    if (columns.empty()) { // There is no extra data
        return;
    }

    columns.emplace_back(sql::col::EVENT_ID);
    values.emplace_back(std::to_string(eventId));

    txn.exec(
        "INSERT INTO " + sql::table::EVENT_EXTRA_DATA + " "
        "(" + common::join(columns, ',') + ") VALUES "
        "(" + common::join(values, ',') + ")"
    );
}

void writeAoiIds(
    pqxx::transaction_base& txn,
    const TIds& aoiIds,
    TId eventId,
    TId branchId)
{
    checkAoi(aoiIds);

    std::vector<std::string> columns {
        sql::col::AOI_ID,
        sql::col::EVENT_ID
    };

    std::vector<std::vector<std::string>> valuesMatrix;
    std::for_each(
        aoiIds.begin(), aoiIds.end(),
        [&](TId aoiId){
            valuesMatrix.push_back({
                std::to_string(aoiId),
                std::to_string(eventId)
            });
        }
    );

    if (branchId) {
        columns.emplace_back(sql::col::BRANCH_ID);

        std::for_each(
            valuesMatrix.begin(), valuesMatrix.end(),
            [&](auto& matrixRow) {
                matrixRow.emplace_back(std::to_string(branchId));
            }
        );
    }

    std::string aoiFeedTable = branchId ?
        sql::table::AOI_FEED_STABLE :
        sql::table::AOI_FEED_TRUNK_WITHOUT_TASK;

    auto insertResult = txn.exec(
        "INSERT INTO " + aoiFeedTable + " (" +
        common::join(columns, ',') + ") VALUES " +
        multirowValuesInsertStatement(valuesMatrix)
    );

    ASSERT(insertResult.affected_rows() == aoiIds.size());
}

std::string feedbackBounds(const feedback::Task& feedback)
{
    auto bboxGeo = geolib3::convertMercatorToGeodetic(
        feedback.position().boundingBox()
    );

    std::ostringstream stream;
    stream.precision(COORDINATE_PRECISION);

    stream << "[" << bboxGeo.minX() << "," << bboxGeo.minY() << ","
                  << bboxGeo.maxX() << "," << bboxGeo.maxY() << "]";

    return stream.str();
}

} // namespace


CommitData::CommitData(
        TId branchId,
        TId commitId,
        std::string action,
        std::string bounds)
    : branchId_(branchId)
    , commitId_(commitId)
    , action_(std::move(action))
    , bounds_(std::move(bounds))
{
    if (!bounds_.empty()) {
        bbox_ = jsonToBbox(bounds_);
    }
}


PrimaryObjectData::PrimaryObjectData(
        TId id,
        std::string categoryId,
        std::string screenLabel,
        std::string editNotes)
    : id_(id)
    , categoryId_(std::move(categoryId))
    , screenLabel_(std::move(screenLabel))
    , editNotes_(std::move(editNotes))
{}


const std::optional<EventExtraData> NO_EVENT_EXTRA_DATA = std::nullopt;

std::string Event::bounds() const
{
    if (commitData_) {
        return commitData_->bounds();
    } else if (feedback_) {
        return feedbackBounds(*feedback_);
    } else {
        REQUIRE(
            false,
            "Task " << id_ << " doesn't contain neither commitData nor feedback"
        );
    }
}

std::optional<std::string>
Event::getPrimaryObjectCategory() const
{
    if (!primaryObjectData_) {
        return std::nullopt;
    }
    const auto& categoryId = primaryObjectData_.value().categoryId();
    return categoryId.empty() ? std::nullopt : std::make_optional(categoryId);
}

Event::Event(
    const pqxx::row& row,
    std::optional<EventExtraData> extraData,
    Kind eventKind
)
    : id_(row[sql::col::EVENT_ID].as<TId>())
    , type_(boost::lexical_cast<EventType>(row[sql::col::TYPE].c_str()))
    , createdBy_(row[sql::col::CREATED_BY].as<TUid>())
    , createdAt_(row[sql::col::CREATED_AT].as<std::string>())
    , action_(row[sql::col::ACTION].as<std::string>())
    , extraData_(std::move(extraData))
{
    if (eventKind == Kind::Commit) {
        commitData_ = CommitData(
            row[sql::col::BRANCH_ID].as<TId>(),
            row[sql::col::COMMIT_ID].as<TId>(),
            row[sql::col::ACTION].as<std::string>(),
            row[sql::col::BOUNDS].as<std::string>({})
        );
        if (!row[sql::col::PRIMARY_OBJECT_ID].is_null()) {
            primaryObjectData_ = PrimaryObjectData(
                row[sql::col::PRIMARY_OBJECT_ID].as<TId>(),
                row[sql::col::PRIMARY_OBJECT_CATEGORY_ID].as<std::string>(),
                row[sql::col::PRIMARY_OBJECT_LABEL].as<std::string>(),
                row[sql::col::PRIMARY_OBJECT_NOTES].as<std::string>()
            );
        }
    }
}

Event
Event::create(
    pqxx::transaction_base& txn,
    EventType type,
    TUid uid,
    const CommitData& commitData,
    const std::optional<PrimaryObjectData>& primaryObjectData,
    const TIds& aoiIds,
    const std::optional<EventExtraData>& extraData)
{
    ASSERT(type != EventType::ClosedFeedback);
    checkUid(uid);

    // Inserting into commit_event
    //
    const auto commitEventInsertResult = [&](){

        std::vector<std::string> columns {
            sql::col::TYPE,
            sql::col::CREATED_BY,
            sql::col::CREATED_AT,
            sql::col::BRANCH_ID,
            sql::col::COMMIT_ID,
            sql::col::ACTION,
            sql::col::BOUNDS
        };

        std::vector<std::string> values {
            txn.quote(boost::lexical_cast<std::string>(type)),
            std::to_string(uid),
            sql::value::NOW,
            std::to_string(commitData.branchId()),
            std::to_string(commitData.commitId()),
            txn.quote(commitData.action()),
            txn.quote(commitData.bounds())
        };

        if (commitData.bbox()) {
            columns.emplace_back(sql::col::BOUNDS_GEOM);
            values.emplace_back([&](){
                const auto& bbox = *commitData.bbox();

                std::ostringstream stream;
                stream.precision(COORDINATE_PRECISION);
                stream << "ST_MakeEnvelope("
                       << bbox.minX() << "," << bbox.minY() << ","
                       << bbox.maxX() << "," << bbox.maxY() << ","
                       << sql::value::MERCATOR_SRID << ")";

                return stream.str();
            }());
        }

        if (primaryObjectData) {
            columns.emplace_back(sql::col::PRIMARY_OBJECT_ID);
            values.emplace_back(std::to_string(primaryObjectData->id()));

            columns.emplace_back(sql::col::PRIMARY_OBJECT_CATEGORY_ID);
            values.emplace_back(txn.quote(primaryObjectData->categoryId()));

            columns.emplace_back(sql::col::PRIMARY_OBJECT_LABEL);
            values.emplace_back(txn.quote(primaryObjectData->screenLabel()));

            columns.emplace_back(sql::col::PRIMARY_OBJECT_NOTES);
            values.emplace_back(txn.quote(primaryObjectData->editNotes()));
        }

        return txn.exec(
            "INSERT INTO " + sql::table::COMMIT_EVENT + " "
            "(" + common::join(columns, ',') + ") VALUES "
            "(" + common::join(values, ',') + ") RETURNING *"
        );
    }();

    ASSERT(commitEventInsertResult.size() == 1);

    // Inserting into event_extra_data
    //
    createEventExtraData(
        txn,
        commitEventInsertResult[0][sql::col::EVENT_ID].as<TId>(),
        extraData);

    Event newEvent(commitEventInsertResult[0], extraData, Kind::Commit);

    // Inserting into aoi_feed_...
    //
    if (!aoiIds.empty()) {
        writeAoiIds(txn, aoiIds, newEvent.id(), commitData.branchId());
    }

    return newEvent;
}

Event
Event::create(
    pqxx::transaction_base& txn,
    TUid uid,
    const std::string& action,
    const feedback::Task& feedbackTask,
    const TIds& aoiIds)
{
    checkUid(uid);
    // Inserting into feedback_event
    //
    auto newEvent = [&](){
        std::vector<std::string> columns {
            sql::col::CREATED_BY,
            sql::col::CREATED_AT,
            sql::col::TYPE,
            sql::col::ACTION,
            sql::col::FEEDBACK_TASK_ID
        };

        std::vector<std::string> values {
            std::to_string(uid),
            sql::value::NOW,
            txn.quote(
                boost::lexical_cast<std::string>(EventType::ClosedFeedback)
            ),
            txn.quote(action),
            std::to_string(feedbackTask.id())
        };

        auto insertedResult = txn.exec(
            "INSERT INTO " + sql::table::FEEDBACK_EVENT + " "
            "(" + common::join(columns, ',') + ") VALUES "
            "(" + common::join(values, ',') + ") RETURNING *"
        );

        ASSERT(insertedResult.size() == 1);

        return Event(
            insertedResult[0],
            std::nullopt,
            Kind::Feedback
        );
    }();

    newEvent.feedback_ = feedbackTask;

    // Inserting into aoi_feed_...
    //
    if (!aoiIds.empty()) {
        writeAoiIds(txn, aoiIds, newEvent.id(), revision::TRUNK_BRANCH_ID);
    }

    return newEvent;
}

} // namespace maps::wiki::social
