#include "feed_helpers.h"
#include "factory.h"

#include <fmt/format.h>
#include <yandex/maps/wiki/common/string_utils.h>

namespace maps::wiki::social {

std::pair<Events, HasMore> createEvents(
    const pqxx::result& r, std::optional<size_t> limit, PushOrder order)
{
    bool more = false;
    Events items;
    for (const auto& row: r) {
        if (limit && items.size() == *limit) {
            more = true;
            break;
        }
        auto eventToInsert = Factory::component<Event>(
            row, NO_EVENT_EXTRA_DATA, Event::Kind::Commit);
        if (order == PushOrder::Back) {
            items.emplace_back(eventToInsert);
        } else {
            items.emplace_front(eventToInsert);
        }
    }
    return {std::move(items), more ? HasMore::Yes : HasMore::No};
}

std::string categoryIds2str(
    pqxx::transaction_base& work, const CategoryIdsSet& categoryIds)
{
    auto category2str = [&](const std::string& categoryId) {
        return work.quote(categoryId);
    };
    return common::join(
        categoryIds.begin(),
        categoryIds.end(),
        category2str,
        ",");
}

std::string categoriesWhereCondition(
    pqxx::transaction_base& work,
    const CategoryIdsFilter& categoryIdsFilter)
{
    bool areIncluded = categoryIdsFilter.policy == InclusionPolicy::Including;
    if (categoryIdsFilter.ids.empty()) {
        return areIncluded ? " FALSE " : " TRUE ";
    }

    std::string formatString =
        areIncluded ? "({column} IN ({catIds}))"
                    : "({column} IS NULL OR {column} NOT IN ({catIds}))";

    return fmt::format(
        formatString,
        fmt::arg("column", sql::col::PRIMARY_OBJECT_CATEGORY_ID),
        fmt::arg("catIds", categoryIds2str(work, categoryIdsFilter.ids)));
}

} // namespace maps::wiki::social
