#include <yandex/maps/wiki/social/feedback/agent.h>
#include <yandex/maps/wiki/social/feedback/attribute_names.h>
#include <yandex/maps/wiki/social/feedback/duplicates.h>
#include <yandex/maps/wiki/social/feedback/task_aoi.h>
#include <yandex/maps/wiki/social/feedback/task_patch.h>

namespace maps::wiki::social::feedback {

namespace {

const RejectReasonsSet FBAPI_REJECT_REASONS = []() {
    const auto& allRejectReasons = enum_io::enumerateValues<RejectReason>();
    RejectReasonsSet reasons(
        allRejectReasons.begin(),
        allRejectReasons.end()
    );
    reasons.erase(RejectReason::NoProcess);
    reasons.erase(RejectReason::NoData);
    return reasons;
}();

const RejectReasonsSet TASK_REJECT_REASONS {
    RejectReason::NoData,
    RejectReason::ProhibitedByRules,
};

const RejectReasonsSet FEEDBACK_REJECT_REASONS {
    RejectReason::IncorrectData,
    RejectReason::NoInfo,
    RejectReason::ProhibitedByRules,
    RejectReason::Spam,
};

const std::set<std::string> SOURCES_FBAPI = {
    "fbapi", "fbapi-nologin", "fbapi-samsara", "experiment-fbapi-samsara"
};

bool canPerformOperation(const Task& task, TUid uid, TaskOperation operation)
{
    return Agent::validOperations(task, uid).contains(operation);
}

bool fromNmaps(const Task& task)
{
    static const std::set<std::string> SOURCES_NMAPS = {
        "nmaps-complaint", "nmaps-request-for-deletion"
    };
    return SOURCES_NMAPS.contains(task.source());
}

bool canChangePosition(const Task& task)
{
    static const std::set<std::string> SOURCES_FOR_CHANGE_POSITION = {
        "fbapi-samsara",
        "experiment-fbapi-samsara",
        "geoq-newsfeed"
    };
    return SOURCES_FOR_CHANGE_POSITION.contains(task.source());
}

// We have separated table for some types of feedback.
// see update trigger: https://a.yandex-team.ru/arc/trunk/arcadia/maps/wikimap/mapspro/migrations/migrations/V440__feedback_update_triger_fixed.sql?rev=7575937#L12
bool isExperimentType(Type type)
{
    static const TypesSet experimentTypes {
        Type::AddressExperiment, Type::EntranceExperiment};
    return experimentTypes.contains(type);
}

TypesSet allCommonTypesSet()
{
    static const TypesSet EXCLUDED_TYPES = {
        Type::RoadClosure,
        Type::OtherHeavy,
    };
    static const TypesSet typesSet = []() {
        Types types = allTypes();
        types.erase(
            std::remove_if(
                types.begin(),
                types.end(),
                [&](Type type) {
                    return isExperimentType(type) || EXCLUDED_TYPES.contains(type);
                }
            ),
            types.end()
        );
        return TypesSet(types.begin(), types.end());
    }();
    return typesSet;
}

} // namespace

Agent::Agent(pqxx::transaction_base& socialTxn, TUid uid)
    : gatewayRw_(socialTxn)
    , uid_(uid)
{}

std::optional<TaskForUpdate> Agent::taskForUpdateById(TId id)
{
    return gatewayRw_.taskForUpdateById(id);
}

TaskForUpdate Agent::addTask(const TaskNew& newTask)
{
    return gatewayRw_.addTask(uid_, newTask);
}

std::optional<TaskForUpdate> Agent::revealTaskByIdCascade(TId id)
{
    const auto task = gatewayRw_.taskForUpdateById(id);
    if (!task) {
        return std::nullopt;
    }
    if (task->revealed()) {
        return task;
    }

    auto headTask = openTask(task.value());
    ASSERT(headTask);

    changeDuplicatesStatusesAccordingToHead(gatewayRw_, uid_, headTask.value());
    return headTask;
}

std::optional<TaskForUpdate> Agent::acquireTask(const TaskForUpdate& task)
{
    if (!canPerformOperation(task, uid_, TaskOperation::Acquire)) {
        return std::nullopt;
    }
    return gatewayRw_.updateTask(task, TaskPatch(uid_).setAcquired());
}

std::optional<TaskForUpdate> Agent::releaseTask(const TaskForUpdate& task)
{
    if (!canPerformOperation(task, uid_, TaskOperation::Release)) {
        return std::nullopt;
    }
    return gatewayRw_.updateTask(task, TaskPatch(uid_).setReleased());
}

std::optional<TaskForUpdate> Agent::changeTaskType(
    const TaskForUpdate& task,
    Type newType)
{
    if (!canPerformOperation(task, uid_, TaskOperation::ChangeType)) {
        return std::nullopt;
    }
    if (!validNewTypes(task).count(newType)) {
        return std::nullopt;
    }
    return gatewayRw_.updateTask(
        task,
        TaskPatch(uid_).setType(newType).setReleased());
}

std::optional<TaskForUpdate> Agent::changeTaskPosition(
    const TaskForUpdate& task,
    const geolib3::Point2& mercPoint,
    const TIds& aoiIdsForNewPosition)
{
    if (!canPerformOperation(task, uid_, TaskOperation::ChangePosition)) {
        return std::nullopt;
    }
    removeTaskFromAoiFeed(gatewayRw_.txn(), task.id());
    addTaskToAoiFeed(gatewayRw_.txn(), task.id(), aoiIdsForNewPosition, getPartition(task));
    return gatewayRw_.updateTask(
        task,
        TaskPatch(uid_).setPosition(mercPoint).setReleased());
}

std::optional<TaskForUpdate> Agent::hideTask(const TaskForUpdate& task)
{
    if (!canPerformOperation(task, uid_, TaskOperation::Hide)) {
        return std::nullopt;
    }
    return gatewayRw_.updateTask(task, TaskPatch(uid_).setHidden(true));
}

std::optional<TaskForUpdate> Agent::showTask(const TaskForUpdate& task)
{
    if (!canPerformOperation(task, uid_, TaskOperation::Show)) {
        return std::nullopt;
    }
    return gatewayRw_.updateTask(task, TaskPatch(uid_).setHidden(false));
}

std::optional<TaskForUpdate> Agent::processingLevelUp(
    const TaskForUpdate& task,
    Verdict suggestedVerdict)
{
    if (!canPerformOperation(task, uid_, TaskOperation::ProcessingLevelUp)) {
        return std::nullopt;
    }
    return gatewayRw_.updateTask(
        task,
        TaskPatch(uid_)
            .setProcessingLvl(ProcessingLvl::Level1, suggestedVerdict)
            .setReleased());
}

std::optional<TaskForUpdate> Agent::changeTaskProcessingLvl(
    const TaskForUpdate& task,
    ProcessingLvl newLvl,
    std::optional<Verdict> suggestedVerdict)
{
    if (!canPerformOperation(task, uid_, TaskOperation::ChangeProcessingLvl)) {
        return std::nullopt;
    }
    if (!validNewProcessingLvls(task).contains(newLvl)) {
        return std::nullopt;
    }
    return gatewayRw_.updateTask(
        task,
        TaskPatch(uid_).setProcessingLvl(newLvl, suggestedVerdict).setReleased());
}

std::optional<TaskForUpdate> Agent::processingLevelDown(const TaskForUpdate& task)
{
    if (!canPerformOperation(task, uid_, TaskOperation::ProcessingLevelDown)) {
        return std::nullopt;
    }
    return gatewayRw_.updateTask(
        task,
        TaskPatch(uid_).setProcessingLvl(ProcessingLvl::Level0));
}

std::optional<TaskForUpdate> Agent::markViewedTask(const TaskForUpdate& task)
{
    if (!canPerformOperation(task, uid_, TaskOperation::MarkViewed)) {
        return std::nullopt;
    }
    if (task.viewedBy().count(uid_)) {
        return task;
    }
    return gatewayRw_.updateTask(task, TaskPatch(uid_).setViewedByUpdater());
}

std::optional<TaskForUpdate> Agent::resolveTaskCascade(
    const TaskForUpdate& task,
    Resolution resolution,
    std::optional<TId> commentId)
{
    if (resolution.verdict() == Verdict::Accepted) {
        if (!canPerformOperation(task, uid_, TaskOperation::Accept)) {
            return std::nullopt;
        }
    } else {
        if (!canPerformOperation(task, uid_, TaskOperation::Reject)) {
            return std::nullopt;
        }
        if (
            resolution.rejectReason() &&
            !validRejectReasons(task).contains(*resolution.rejectReason())
        ) {
            return std::nullopt;
        }
    }

    TaskPatch patch(uid_);
    patch.setResolution(resolution)
        .setBucket(Bucket::Outgoing)
        .setReleased();
    if (commentId) {
        patch.setCommentId(*commentId);
    }
    auto headTask = gatewayRw_.updateTask(task, patch);

    changeDuplicatesStatusesAccordingToHead(gatewayRw_, uid_, headTask);
    return headTask;
}

std::optional<TaskForUpdate> Agent::openTask(
    const TaskForUpdate& task,
    std::optional<TId> commentId)
{
    if (!canPerformOperation(task, uid_, TaskOperation::Open)) {
        return std::nullopt;
    }

    auto patch = TaskPatch(uid_);
    patch.setBucket(Bucket::Outgoing)
        .setReleased()
        .setResolution(std::nullopt);
    if (commentId) {
        patch.setCommentId(commentId.value());
    }
    return gatewayRw_.updateTask(task, patch);
}

std::optional<TaskForUpdate> Agent::needInfoTask(
    const TaskForUpdate& task,
    TId commentId,
    std::optional<std::string> requestTemplate)
{
    if (!canPerformOperation(task, uid_, TaskOperation::NeedInfo)) {
        return std::nullopt;
    }

    auto patch = TaskPatch(uid_);
    patch.setNeedInfo(requestTemplate)
        .setReleased()
        .setResolution(std::nullopt)
        .setCommentId(commentId);

    auto headTask = gatewayRw_.updateTask(task, patch);
    return headTask;
}

std::optional<TaskForUpdate> Agent::commentTask(
    TId taskId,
    TId commentId)
{
    return gatewayRw_.updateTaskById(
        taskId,
        TaskPatch(uid_).setCommentId(commentId));
}

std::optional<TaskForUpdate> Agent::deployTaskByIdCascade(
    TId id,
    chrono::TimePoint at)
{
    auto task = gatewayRw_.taskForUpdateById(id);
    if (!task) {
        return std::nullopt;
    }
    if (task->deployedAt()) {
        return task;
    }

    auto headTask = gatewayRw_.updateTask(*task,
        TaskPatch(uid_).setDeployedAt(at));

    changeDuplicatesStatusesAccordingToHead(gatewayRw_, uid_, headTask);
    return headTask;
}

RejectReasonsSet Agent::validRejectReasons(const Task& task)
{
    if (task.workflow() == Workflow::Task) {
        return TASK_REJECT_REASONS;
    }
    if (fromFbapi(task)) {
        return FBAPI_REJECT_REASONS;
    }
    return FEEDBACK_REJECT_REASONS;
}

ProcessingLvlSet Agent::validNewProcessingLvls(const Task& task)
{
    switch (task.processingLvl()) {
        case ProcessingLvl::Level0: {
            return ProcessingLvlSet{
                ProcessingLvl::Level1,
                ProcessingLvl::Level2,
                ProcessingLvl::BigTask,
            };
        }
        case ProcessingLvl::Level1: {
            return ProcessingLvlSet{
                ProcessingLvl::Level0,
                ProcessingLvl::Level2,
                ProcessingLvl::BigTask,
            };
        }
        case ProcessingLvl::Level2: {
            return ProcessingLvlSet{
                ProcessingLvl::BigTask,
            };
        }
        case ProcessingLvl::BigTask: {
            return ProcessingLvlSet{
                ProcessingLvl::Level2,
            };
        }
    }
}

TypesSet Agent::validNewTypes(const Task& task)
{
    if (isExperimentType(task.type())) {
        return TypesSet{};
    } else if (task.type() == Type::RoadClosure) {
        return TypesSet{
            Type::Road,
            Type::StreetName,
            Type::RoadDirection,
            Type::PublicTransportStop,
            Type::Parking,
            Type::Maneuver,
            Type::Barrier,
            Type::TrafficLight,
            Type::Other,
        };
    } else if (task.workflow() == Workflow::Feedback) {
        TypesSet typesSet = allCommonTypesSet();
        typesSet.erase(task.type());
        if (fromNmaps(task) && task.type() != Type::OtherHeavy) {
            typesSet.emplace(Type::OtherHeavy);
        }
        return typesSet;
    } else {
        return TypesSet{};
    }
}

TaskOperations Agent::validOperations(const Task& task, TUid uid)
{
    TaskOperations retVal;

    switch (task.state()) {
        case TaskState::Incoming: {
            retVal.insert(TaskOperation::Defer);
            retVal.insert(TaskOperation::Open);
            break;
        }
        case TaskState::Deferred: {
            retVal.insert(TaskOperation::Open);
            break;
        }
        case TaskState::Opened: {
            if (canRequestInfo(task)) {
                retVal.insert(TaskOperation::NeedInfo);
            }
            if (task.acquired()) {
                if (task.acquired()->uid == uid) {
                    retVal.insert(TaskOperation::Release);
                    retVal.insert(TaskOperation::Acquire);
                    if (canChangePosition(task)) {
                        retVal.insert(TaskOperation::ChangePosition);
                    }
                }
            } else {
                retVal.insert(TaskOperation::Acquire);
            }
            retVal.insert(TaskOperation::Accept);
            retVal.insert(TaskOperation::Reject);
            if (!validNewTypes(task).empty()) {
                retVal.insert(TaskOperation::ChangeType);
            }
            if (!validNewProcessingLvls(task).empty()) {
                retVal.insert(TaskOperation::ChangeProcessingLvl);
            }
            break;
        }
        case TaskState::NeedInfo: {
            retVal.insert(TaskOperation::Open);
            break;
        }
        case TaskState::Rejected: {
            if (canRequestInfo(task)) {
                retVal.insert(TaskOperation::NeedInfo);
            }
            if (!task.duplicateHeadId()) {
                retVal.insert(TaskOperation::Open);
            }
            retVal.insert(TaskOperation::Reject);
            break;
        }
        case TaskState::Accepted: {
            if (canRequestInfo(task)) {
                retVal.insert(TaskOperation::NeedInfo);
            }
            if (!task.duplicateHeadId()) {
                retVal.insert(TaskOperation::Open);
            }
            retVal.insert(TaskOperation::Accept);
            retVal.insert(TaskOperation::Deploy);
            break;
        }
        case TaskState::Deployed: {
            // terminal state
            break;
        }
    }

    if (!task.acquired() || task.acquired()->uid == uid) {
        retVal.insert(TaskOperation::Hide);
    }

    retVal.insert(TaskOperation::Comment);
    retVal.insert(TaskOperation::MarkViewed);

    if (task.processingLevel() < PROCESSING_LEVEL_MAX) {
        retVal.insert(TaskOperation::ProcessingLevelUp);
    }
    if (task.processingLevel() > PROCESSING_LEVEL_MIN) {
        retVal.insert(TaskOperation::ProcessingLevelDown);
    }

    return retVal;
}

bool Agent::fromFbapi(const Task& task)
{
    return SOURCES_FBAPI.contains(task.source());
}

bool Agent::canRequestInfo(const Task& task)
{
    const auto& attrs = task.attrs();
    return fromFbapi(task)
        && attrs.existCustom(attrs::USER_EMAIL)
        && !attrs.getFlag(attrs::FROM_PUSH);
}

} // namespace maps::wiki::social::feedback
