#include <yandex/maps/wiki/social/feedback/description_producers.h>
#include <yandex/maps/wiki/social/feedback/description.h>
#include <yandex/maps/wiki/social/feedback/description_keys.h>
#include <yandex/maps/wiki/social/feedback/enums.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <maps/libs/common/include/exception.h>

#include <sstream>

namespace maps::wiki::social::feedback {

using RoadDirection = ymapsdf::rd::Direction;

namespace {

std::map<RoadDirection, std::string> DIR_TO_ATTR {
    {RoadDirection::Forward,  tanker::attr_values::RDEL_ONEWAY_FROM_KEY},
    {RoadDirection::Backward, tanker::attr_values::RDEL_ONEWAY_TO_KEY},
    {RoadDirection::Both,     tanker::attr_values::RDEL_ONEWAY_BOTH_KEY},
};

} // namespace anonymous

Description AbsentObjectDescr::toDescription() const
{
    ParamToDescription params;
    if (correctName) {
        params[tanker::CORRECT_NAME] = correctName.value();
    }

    return DescriptionI18n(
        tanker::fb_desc::ABSENT_OBJECT_KEY,
        std::move(params)
    );
}

Description AbsentRoadDescr::toDescription() const
{
    ParamToDescription params;
    if (roadName) {
        params[tanker::ROAD_NAME] = roadName.value();
    }

    return DescriptionI18n(
        tanker::fb_desc::ABSENT_ROAD_KEY,
        std::move(params)
    );
}

Description WrongRoadDirectionDescr::toDescription() const
{
    ParamToDescription params{
        {tanker::CORRECT_ROAD_DIR,
         DescriptionI18n(DIR_TO_ATTR.at(correctDirection), {})}
    };

    return DescriptionI18n(
        tanker::fb_desc::WRONG_ROAD_DIR_KEY, std::move(params));
}

Description WrongSpeedLimitDescr::toDescription() const
{
    ParamToDescription params;
    if (direction) {
        params[tanker::DIRECTION] = DescriptionI18n(DIR_TO_ATTR.at(*direction), {});
    }
    if (truck) {
        params[tanker::TRUCK] = std::string("true");
    }
    params[tanker::CORRECT_SPEED_LIMIT] = std::to_string(correctSpeedLimit);
    if (currentSpeedLimit) {
        params[tanker::CURRENT_SPEED_LIMIT] = std::to_string(*currentSpeedLimit);
    }
    if (confirmingSignNotFound) {
        params[tanker::CONFIRMING_SIGN_NOT_FOUND] = std::string("true");
    }

    return DescriptionI18n(
        tanker::fb_desc::WRONG_SPEEDLIMIT_KEY, std::move(params));
}

Description WrongNameDescr::toDescription() const
{
    ParamToDescription params;
    if (currentName) {
        params[tanker::CURRENT_NAME] = *currentName;
    }
    if (correctName) {
        params[tanker::CORRECT_NAME] = *correctName;
    }

    return DescriptionI18n(
        tanker::fb_desc::WRONG_NAME_KEY,
        std::move(params)
    );
}

namespace {

using FtType = ymapsdf::ft::Type;

const std::map<FtType, std::string> parkingTypeToStr {
    {FtType::UrbanRoadnetParkingFree, "'бесплатная'"},
    {FtType::UrbanRoadnetParkingToll, "'платная'"},
    {FtType::UrbanRoadnetParkingRestricted, "'ограничена'"},
    {FtType::UrbanRoadnetParkingProhibited, "'запрещена'"},
};

const std::map<FtType, std::string> parkingTypeToAttrValue {
    {FtType::UrbanRoadnetParkingFree,
        tanker::attr_values::LINEAR_PARKING_FREE_KEY},
    {FtType::UrbanRoadnetParkingToll,
        tanker::attr_values::LINEAR_PARKING_TOLL_KEY},
    {FtType::UrbanRoadnetParkingRestricted,
        tanker::attr_values::LINEAR_PARKING_RESTRICT_KEY},
    {FtType::UrbanRoadnetParkingProhibited,
        tanker::attr_values::LINEAR_PARKING_PROHIBIT_KEY}
};

} // namespace anonymous

Description WrongParkingTypeDescr::toDescription() const
{
    ParamToDescription params {
        {tanker::CURRENT_PARKING_TYPE,
         DescriptionI18n(parkingTypeToAttrValue.at(currentParkingType), {})},
        {tanker::CORRECT_PARKING_TYPE,
         DescriptionI18n(parkingTypeToAttrValue.at(correctParkingType), {})}
    };

    return DescriptionI18n(
        tanker::fb_desc::WRONG_PARKING_TYPE_KEY, std::move(params));
}

Description AbsentEntranceDescr::toDescription() const
{
    ParamToDescription params;
    if (entranceName) {
        params[tanker::ENTRANCE_NAME] = *entranceName;
    }

    return DescriptionI18n(
        tanker::fb_desc::ABSENT_ENTRANCE_KEY, std::move(params));
}

Description DeleteEntranceDescr::toDescription() const
{
    ParamToDescription params;
    if (entranceName) {
        params[tanker::ENTRANCE_NAME] = *entranceName;
    }

    return DescriptionI18n(
        tanker::fb_desc::DELETE_ENTRANCE_KEY, std::move(params));
}

Description EntranceWrongPositionDescr::toDescription() const
{
    ParamToDescription params;
    if (entranceName) {
        params[tanker::ENTRANCE_NAME] = *entranceName;
    }

    return DescriptionI18n(
        tanker::fb_desc::ENTRANCE_WRONG_POS_KEY, std::move(params));
}

Description EntranceCorrectionDescr::toDescription() const
{
    ParamToDescription params;
    if (entranceName) {
        params[tanker::ENTRANCE_NAME] = *entranceName;
    }
    if (correctName) {
        params[tanker::CORRECT_NAME] = *correctName;
    }

    return DescriptionI18n(
        tanker::fb_desc::ENTRANCE_CORRECTION_KEY,
        std::move(params)
    );
}

Description SubwayCorrectionNameAndPositionDescr::toDescription() const
{
    ParamToDescription params;
    params[tanker::CORRECT_NAME] = correctName;

    return DescriptionI18n(
        tanker::fb_desc::SUBWAY_WRONG_NAME_AND_POSITION_KEY,
        std::move(params)
    );
}

Description SubwayCorrectionNameDescr::toDescription() const
{
    ParamToDescription params;
    params[tanker::CORRECT_NAME] = correctName;

    return DescriptionI18n(
        tanker::fb_desc::SUBWAY_WRONG_NAME_KEY,
        std::move(params)
    );
}

Description RemoveObjectDescr::toDescription() const
{
    ParamToDescription params;
    if (name) {
        params[tanker::NAME] = *name;
    }

    return DescriptionI18n(
        tanker::fb_desc::REMOVE_OBJECT_KEY,
        std::move(params)
    );
}

Description AddressPositionCorrectionDescr::toDescription() const
{
    ParamToDescription params{
        {tanker::X_WRONG, std::to_string(geoPosWrong.x())},
        {tanker::Y_WRONG, std::to_string(geoPosWrong.y())},
        {tanker::X_CORRECT, std::to_string(geoPosCorrect.x())},
        {tanker::Y_CORRECT, std::to_string(geoPosCorrect.y())}
    };

    return DescriptionI18n(
        tanker::fb_desc::ADDR_POS_CORRECTION_KEY, std::move(params));
}

Description AddressDescr::toDescription() const
{
    ParamToDescription params;

    if (correctStreet) {
        params[tanker::CORRECT_STREET] = correctStreet.value();
    }
    if (correctHouse) {
        params[tanker::CORRECT_HOUSE] = correctHouse.value();
    }
    if (correctAddressString) {
        params[tanker::CORRECT_ADDRESS] = correctAddressString.value();
    }

    std::string tankerKey;
    switch (type) {
        case AddressDescriptionType::NewAddress:
            tankerKey = tanker::fb_desc::NEW_ADDRESS_KEY;
            break;

        case AddressDescriptionType::AddressCorrection:
            tankerKey = tanker::fb_desc::ADDR_CORRECTION_KEY;
            break;
    }
    return DescriptionI18n(tankerKey, std::move(params));
}

Description NoPublicTransportStopDescr::toDescription() const
{
    ParamToDescription params;
    if (publicStopName) {
        params[tanker::NAME] = publicStopName.value();
    }
    return DescriptionI18n(
        tanker::fb_desc::NO_PUBLIC_TRANSPORT_STOP_HERE_KEY,
        std::move(params)
    );
}

Description RouteGapDescr::toDescription() const
{
    ParamToDescription params;
    params[tanker::COUNT_TRACKS] = std::to_string(tracksCount);
    params[tanker::PROBABILITY] = std::to_string(probability);
    params[tanker::COUNT_TRACK_RIPS] = std::to_string(countTrackRips);
    params[tanker::ROUTE_GAP_LINK] = link;

    return DescriptionI18n(
        tanker::fb_desc::ROUTE_GAP_KEY,
        std::move(params)
    );
}

Description NoNeededParkingTypeDescr::toDescription() const
{
    if (!toll.has_value()) {
        return DescriptionI18n(tanker::fb_desc::NO_NEEDED_PARKING_TYPE_KEY, {});
    }
    if (changeExistedObject) {
        if (toll.value()) {
            return DescriptionI18n(tanker::fb_desc::CHANGE_PARKING_TO_TOLL_KEY, {});
        }
        return DescriptionI18n(tanker::fb_desc::CHANGE_PARKING_TO_FREE_KEY, {});
    }
    if (toll.value()) {
        return DescriptionI18n(tanker::fb_desc::NO_NEEDED_PARKING_TOLL_KEY, {});
    }
    return DescriptionI18n(tanker::fb_desc::NO_NEEDED_PARKING_FREE_KEY, {});
}

Description MandatoryDirectionTrafficSignDescr::toDescription()
{
    return DescriptionI18n(tanker::fb_desc::MANDATORY_DIR_TRAFFIC_SIGN_KEY, {});
}

Description ProhibitedTurnSignDescr::toDescription()
{
    return DescriptionI18n(tanker::fb_desc::PROHIBITED_TURN_SIGN_KEY, {});
}

Description OneWayTrafficSignDescr::toDescription()
{
    return DescriptionI18n(tanker::fb_desc::ONEWAY_TRAFFIC_SIGN_KEY, {});
}

Description TrafficLaneSignDescr::toDescription()
{
    return DescriptionI18n(tanker::fb_desc::TRAFFIC_LANE_SIGN_KEY, {});
}

Description TrafficProhibitedSignDescr::toDescription()
{
    return DescriptionI18n(tanker::fb_desc::TRAFFIC_PROHIBITED_SIGN_KEY, {});
}

Description TrafficCircleSignDescr::toDescription()
{
    return DescriptionI18n(tanker::fb_desc::TRAFFIC_CIRCLE_SIGN_KEY, {});
}

namespace {

std::map<SuggestedAction, std::string> SUGGESTED_ACTION_TO_ATTR {
    {SuggestedAction::Modify,  tanker::fb_desc::SUGGESTED_ACTION_MODIFY_KEY},
    {SuggestedAction::Delete, tanker::fb_desc::SUGGESTED_ACTION_DELETE_KEY},
    {SuggestedAction::CreateIndoorPoi, tanker::fb_desc::INDOOR_POI_MISSED_KEY},
    {SuggestedAction::CreatePoi, tanker::fb_desc::SUGGESTED_ACTION_CREATE_POI_KEY},
    {SuggestedAction::VerifyPosition, tanker::fb_desc::SUGGESTED_ACTION_VERIFY_POSITION_KEY},
};

} // namespace anonymous

Description SuggestedActionDescr::toDescription() const
{
    return DescriptionI18n(SUGGESTED_ACTION_TO_ATTR.at(action), {});
}

namespace {

const std::string& getDescriptionKey(Movement movement)
{
    static const std::unordered_map<Movement, const std::string&> MOVEMENT_TO_DESCRIPTION {
        {Movement::Forward, tanker::fb_desc::PROHIBITED_FORWARD_MOVEMENT},
        {Movement::RightTurn, tanker::fb_desc::PROHIBITED_RIGHT_TURN},
        {Movement::LeftTurn, tanker::fb_desc::PROHIBITED_LEFT_TURN},
        {Movement::Uturn, tanker::fb_desc::PROHIBITED_UTURN},
    };

    const auto it = MOVEMENT_TO_DESCRIPTION.find(movement);
    REQUIRE(
        it != MOVEMENT_TO_DESCRIPTION.end(),
        "Impossible find description for movement '" << movement  << "'"
    );

    return it->second;
}

} // namespace

Description ProhibitedPathDescr::toDescription() const
{
    ParamToDescription params;
    params[tanker::PROHIBITED_PATH_URL] = prohibitedPathUrl;
    if (movement) {
        return DescriptionI18n(
            getDescriptionKey(*movement), std::move(params));
    }
    if (prohibited){
        params[tanker::PROHIBITED] = std::string("true");
    }
    if (accessId == maps::ymapsdf::rd::AccessId::Truck) {
        params[tanker::TRUCK] = std::string("true");
    }

    return DescriptionI18n(
        tanker::fb_desc::PROHIBITED_PATH, std::move(params));

}

Description AbsentAddressDescr::toDescription() const
{
    ParamToDescription params{
        {tanker::ADDRESS_HOUSE_NUMBER, number}
    };
    return DescriptionI18n(tanker::fb_desc::ABSENT_ADDRESS_KEY, std::move(params));
}

Description AbsentTrafficLightDescr::toDescription()
{
    return DescriptionI18n(tanker::fb_desc::ABSENT_TRAFFIC_LIGHT_KEY, {});
}

Description SpeedBumpSignDescr::toDescription() const
{
    if (ahead) {
        return DescriptionI18n(tanker::fb_desc::SPEED_BUMP_SIGN_AHEAD_KEY, {});
    }

    return DescriptionI18n(tanker::fb_desc::SPEED_BUMP_SIGN_KEY, {});
}

} // namespace maps::wiki::social::feedback
