#include <yandex/maps/wiki/social/feedback/description_serialize.h>
#include <yandex/maps/wiki/social/feedback/description.h>
#include <maps/libs/common/include/exception.h>

namespace maps::wiki::social::feedback {

namespace {

template<typename T>
std::string toJson(const T& desc)
{
    json::Builder builder;
    json(desc, json::VariantBuilder(&builder));
    return builder.str();
}

} // namespace anonymous

template<>
DescriptionI18n fromJson(const json::Value& json)
{
    ASSERT(json.isObject());
    ASSERT(json.hasField(I18N_KEY));
    ASSERT(json.hasField(I18N_PARAMS));
    ASSERT(json.fields().size() == 2);

    std::string i18nKey = json[I18N_KEY].as<std::string>();
    ParamToDescription i18nParams;

    const auto& paramsVal = json[I18N_PARAMS];
    for (const auto& param : paramsVal.fields()) {
        i18nParams[param] = fromJson<Description>(paramsVal[param]);
    }

    return DescriptionI18n(std::move(i18nKey), std::move(i18nParams));
}

template<>
Description fromJson(const json::Value& json)
{
    if (json.isString()) {
        return json.toString();
    } else if (json.isObject()) {
        return fromJson<DescriptionI18n>(json);
    } else {
        throw LogicError() <<
            "Description json is neither string nor object";
    }
}

void json(const Description& desc, json::VariantBuilder builder)
{
    if (desc.isNonTranslatable()) {
        builder << desc.asNonTranslatable();
    } else {
        json(desc.asTranslatable(), builder);
    }
}

void json(const DescriptionI18n& desc, json::VariantBuilder builder)
{
    builder << [&](json::ObjectBuilder objBuilder){
        objBuilder[I18N_KEY] << desc.i18nKey();
        objBuilder[I18N_PARAMS] << [&](json::ObjectBuilder paramBuilder) {
            for (const auto& [param, value] : desc.i18nParams()) {
                paramBuilder[param] << value;
            }
        };
    };
}

std::string toJson(const Description& desc)
{
    return toJson<Description>(desc);
}

std::string toJson(const DescriptionI18n& desc)
{
    return toJson<DescriptionI18n>(desc);
}

} // namespace maps::wiki::social::feedback
