#include <maps/wikimap/mapspro/libs/social/magic_strings.h>
#include <maps/libs/geolib/include/conversion.h>
#include <yandex/maps/wiki/social/feedback/history_patch.h>
#include <yandex/maps/wiki/social/feedback/task.h>
#include <yandex/maps/wiki/social/feedback/task_patch.h>

#include <iomanip>
#include <ios>
#include <sstream>

namespace maps::wiki::social::feedback {

namespace {

std::string asString(const geolib3::Point2& point)
{
    std::stringstream ss;
    ss << std::fixed << std::setprecision(9) << point.x() << "," << point.y();
    return ss.str();
}

}  // unnamed namespace

HistoryItemNew::HistoryItemNew(
    TUid modifiedBy,
    TaskOperation operation,
    HistoryItemParams params,
    std::optional<TId> commentId)
    : modifiedBy_(modifiedBy)
    , operation_(operation)
    , params_(std::move(params))
    , commentId_(commentId)
{
    ASSERT(operation != TaskOperation::Unrecognized);
}

HistoryItemNew HistoryItemNew::create(
    TUid modifiedBy,
    TaskOperation operation,
    std::optional<TId> commentId)
{
    std::set<TaskOperation> OPERATIONS_WITH_PARAMS {
        TaskOperation::ChangeType,
        TaskOperation::ChangePosition,
        TaskOperation::Reject,
        TaskOperation::NeedInfo,
        TaskOperation::ProcessingLevelUp,
        TaskOperation::ChangeProcessingLvl,
    };
    ASSERT(!OPERATIONS_WITH_PARAMS.contains(operation));

    return HistoryItemNew(
        modifiedBy,
        operation,
        HistoryItemParams(),
        commentId);
}

HistoryItemNew HistoryItemNew::createProcessingLevelUp(
    TUid modifiedBy,
    Verdict suggestedVerdict)
{
    HistoryItemParams params;
    params[HISTORY_PARAM_SUGGESTED_VERDICT] = toString(suggestedVerdict);
    return HistoryItemNew(modifiedBy, TaskOperation::ProcessingLevelUp, std::move(params));
}

HistoryItemNew HistoryItemNew::createRejected(
    TUid modifiedBy,
    std::optional<RejectReason> rejectReason,
    std::optional<TId> commentId)
{
    HistoryItemParams params;
    if (rejectReason) {
        params[HISTORY_PARAM_REJECT_REASON] = toString(*rejectReason);
    }
    return HistoryItemNew(modifiedBy, TaskOperation::Reject, std::move(params), commentId);
}

HistoryItemNew HistoryItemNew::createChangeProcessingLvl(
    TUid modifiedBy,
    ProcessingLvl newLvl,
    std::optional<Verdict> suggestedVerdict)
{
    HistoryItemParams params;
    params[HISTORY_PARAM_NEW_PROCESSING_LVL] = toString(newLvl);
    if (suggestedVerdict) {
        params[HISTORY_PARAM_SUGGESTED_VERDICT] = toString(*suggestedVerdict);
    }
    return HistoryItemNew(modifiedBy, TaskOperation::ChangeProcessingLvl, std::move(params));
}

HistoryItemNew HistoryItemNew::createNeedInfo(
    TUid modifiedBy,
    std::optional<std::string> requestTemplate,
    std::optional<TId> commentId)
{
    HistoryItemParams params;
    if (requestTemplate) {
        params[HISTORY_PARAM_TEMPLATE] = std::move(*requestTemplate);
    }
    return HistoryItemNew(
        modifiedBy, TaskOperation::NeedInfo, std::move(params), commentId);
}

HistoryItemNew HistoryItemNew::createChangeType(
    TUid modifiedBy,
    Type oldType,
    Type newType)
{
    HistoryItemParams params;
    params[HISTORY_PARAM_OLD_TYPE] = toString(oldType);
    params[HISTORY_PARAM_NEW_TYPE] = toString(newType);
    return HistoryItemNew(modifiedBy, TaskOperation::ChangeType, std::move(params));
}

HistoryItemNew HistoryItemNew::createChangePosition(
    TUid modifiedBy,
    const geolib3::Point2& oldPositionGeo,
    const geolib3::Point2& newPositionGeo)
{
    HistoryItemParams params;
    params[HISTORY_PARAM_OLD_POSITION] = asString(oldPositionGeo);
    params[HISTORY_PARAM_NEW_POSITION] = asString(newPositionGeo);
    return HistoryItemNew(modifiedBy, TaskOperation::ChangePosition, std::move(params));
}


HistoryPatch::HistoryPatch(std::vector<HistoryItemNew> items)
    : items_(std::move(items))
{}

HistoryPatch HistoryPatch::patchOnCreate(TUid uid)
{
    std::vector<HistoryItemNew> items;
    items.push_back(HistoryItemNew::create(uid, TaskOperation::Create));
    return HistoryPatch(items);
}

HistoryPatch HistoryPatch::patchOnRelease(TUid uid)
{
    std::vector<HistoryItemNew> items;
    items.push_back(HistoryItemNew::create(uid, TaskOperation::Release));
    return HistoryPatch(items);
}

HistoryPatch HistoryPatch::patchEmpty()
{
    std::vector<HistoryItemNew> items;
    return HistoryPatch(items);
}

namespace {

bool isRejectReasonChanged(const TaskForUpdate& task, const TaskPatch& patch)
{
    ASSERT(task.resolved());

    return patch.resolution() && patch.resolution().value() &&
           (patch.resolution().value()->rejectReason() !=
            task.resolved()->resolution.rejectReason());
}

} // namespace

HistoryPatch HistoryPatch::patchOnUpdate(
    const TaskForUpdate& task,
    const TaskPatch& patch)
{
    auto uid = patch.updater();
    std::vector<HistoryItemNew> items;
    std::optional<TaskOperation> changeStatusOp;

    // No HistoryItem for changing duplicateHeadId.
    // No HistoryItem for changing viewedBy.

    if (patch.processingLvl()) {
        auto newProcessingLvl = patch.processingLvl().value();
        if (newProcessingLvl != task.processingLvl()) {
            items.push_back(HistoryItemNew::createChangeProcessingLvl(
                uid, newProcessingLvl, patch.suggestedVerdict()));
        }
    }

    auto newState = task.state(patch);
    if (task.state() != newState) {
        switch (newState) {
            case TaskState::Incoming:
                REQUIRE(false, "Invalid feedback modification");
                break;

            case TaskState::Deferred:
                changeStatusOp = TaskOperation::Defer;
                break;

            case TaskState::Opened:
                changeStatusOp = task.revealed()
                    ? TaskOperation::Open
                    : TaskOperation::Reveal;
                break;

            case TaskState::NeedInfo:
                changeStatusOp = TaskOperation::NeedInfo;
                break;

            case TaskState::Accepted:
                changeStatusOp = TaskOperation::Accept;
                break;

            case TaskState::Rejected:
                changeStatusOp = TaskOperation::Reject;
                break;

            case TaskState::Deployed:
                changeStatusOp = TaskOperation::Deploy;
                break;
        }
    }

    if (patch.isAcquired()) {
        if (patch.isAcquired().value()) {
            items.push_back(HistoryItemNew::create(uid, TaskOperation::Acquire));
        } else {
            if (task.acquired()) {
                items.push_back(HistoryItemNew::create(uid, TaskOperation::Release));
            }
        }
    }

    if (patch.hidden()) {
        if (patch.hidden().value()) {
            if (!task.hidden()) {
                items.push_back(HistoryItemNew::create(uid, TaskOperation::Hide));
            }
        } else {
            if (task.hidden()) {
                items.push_back(HistoryItemNew::create(uid, TaskOperation::Show));
            }
        }
    }

    if (patch.type()) {
        if (patch.type().value() != task.type()) {
            items.push_back(HistoryItemNew::createChangeType(uid, task.type(), patch.type().value()));
        }
    }

    if (patch.position()) {
        items.push_back(HistoryItemNew::createChangePosition(
            uid,
            geolib3::convertMercatorToGeodetic(task.position()),
            geolib3::convertMercatorToGeodetic(patch.position().value())));
    }

    std::optional<RejectReason> rejectReasonOpt;
    if (patch.resolution() && patch.resolution().value()) {
        rejectReasonOpt = patch.resolution().value()->rejectReason();
    }

    if (changeStatusOp) {
        if (*changeStatusOp == TaskOperation::Reject) {
            items.push_back(HistoryItemNew::createRejected(uid, rejectReasonOpt, patch.commentId()));
        } else if (*changeStatusOp == TaskOperation::NeedInfo) {
            items.push_back(HistoryItemNew::createNeedInfo(
                uid, patch.needInfoRequestTemplate(), patch.commentId()));
        } else {
            items.push_back(HistoryItemNew::create(uid, *changeStatusOp, patch.commentId()));
        }
    } else if (task.resolved() && isRejectReasonChanged(task, patch)) {
        ASSERT(task.state() == TaskState::Rejected);
        items.push_back(HistoryItemNew::createRejected(uid, rejectReasonOpt, patch.commentId()));
    } else if (patch.commentId()) {
        items.push_back(HistoryItemNew::create(uid, TaskOperation::Comment, patch.commentId()));
    }

    return HistoryPatch(items);
}

} //namespace maps::wiki::social::feedback
