#include <yandex/maps/wiki/social/feedback/preset.h>
#include <yandex/maps/wiki/social/feedback/task_filter.h>
#include <yandex/maps/wiki/social/feedback/enums.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/common/pg_utils.h>
#include <maps/libs/enum_io/include/enum_io.h>
#include <maps/libs/json/include/builder.h>
#include <maps/libs/json/include/value.h>

#include <maps/wikimap/mapspro/libs/social/magic_strings.h>
#include <maps/wikimap/mapspro/libs/social/helpers.h>

#include <algorithm>
#include <boost/lexical_cast.hpp>
#include <boost/range/adaptor/map.hpp>
#include <sstream>

namespace maps::wiki::social::feedback {

namespace {

const enum_io::Representations<RoleKind> ROLE_KIND_STRINGS {
    {RoleKind::Read, "Read"},
    {RoleKind::AllRights, "AllRights"},
};

const std::map<RoleKind, std::string> ROLE_KIND_TO_COL_NAME {
    {RoleKind::Read, sql::col::READ_ROLE_ID},
    {RoleKind::AllRights, sql::col::ALL_RIGHTS_ROLE_ID},
};

template<typename T>
std::string toSqlArray(const std::vector<T>& array, pqxx::transaction_base& txn)
{
    std::vector<std::string> strings;
    std::transform(
        array.begin(), array.end(),
        std::back_inserter(strings),
        [](const T& val) {
            return boost::lexical_cast<std::string>(val);
        }
    );
    return txn.quote('{' + common::join(strings, ",") + '}');
}

template<typename T>
std::string toSqlArrayOrNull(
    const std::optional<std::vector<T>>& array,
    pqxx::transaction_base& txn)
{
    return array.has_value() ? toSqlArray(array.value(), txn) : "NULL";
}

template<typename T>
std::string toQuotedString(
    pqxx::transaction_base& txn,
    const T& value)
{
    std::string stringValue = boost::lexical_cast<std::string>(value);
    return txn.quote(stringValue);
}

std::string
toSqlJsonArrayOrNull(
    const std::optional<std::vector<std::string>>& sources,
    pqxx::transaction_base& txn)
{
    if (!sources.has_value()) {
        return "NULL";
    }

    std::vector<json::Value> fields;

    std::transform(
        sources->begin(), sources->end(),
        std::back_inserter(fields),
        [&](const std::string& source) {
            return json::Value(source);
        }
    );

    return txn.quote(
        (json::Builder() << json::Value(std::move(fields))).str()
    );
}

std::vector<std::string>
jsonArrayToVector(const json::Value& jsonArray)
{
    ASSERT(jsonArray.isArray());

    std::vector<std::string> array;

    std::transform(
        jsonArray.begin(), jsonArray.end(),
        std::back_inserter(array),
        [&](const json::Value& json) {
            return json.as<std::string>();
        }
    );

    return array;
}

const std::string PRESET_FIELDS = common::join(
    std::vector<std::string> {
        sql::col::ID,
        sql::col::NAME,
        sql::col::TYPES,
        sql::col::WORKFLOWS,
        sql::col::SOURCES,
        sql::col::AGE_TYPES,
        sql::col::HIDDEN,
        sql::col::STATUS,
        sql::col::READ_ROLE_ID,
        sql::col::ALL_RIGHTS_ROLE_ID,
    },
    ","
);


Preset presetFromPqxxRow(const pqxx::row& row)
{
    Preset preset;

    preset.id = row[sql::col::ID].as<TId>();
    preset.name = row[sql::col::NAME].as<std::string>();
    for (auto kind : enum_io::enumerateValues<RoleKind>()) {
        auto colName = ROLE_KIND_TO_COL_NAME.at(kind);
        preset.roles.setId(kind, row[colName].as<TId>());
    }

    if (!row[sql::col::TYPES].is_null()) {
        preset.entries.types = common::parseSqlArray<Type>(
            row[sql::col::TYPES].as<std::string>()
        );
    }

    if (!row[sql::col::WORKFLOWS].is_null()) {
        preset.entries.workflows = common::parseSqlArray<Workflow>(
            row[sql::col::WORKFLOWS].as<std::string>()
        );
    }

    if (!row[sql::col::SOURCES].is_null()) {
        preset.entries.sources = jsonArrayToVector(
            json::Value::fromString(row[sql::col::SOURCES].as<std::string>())
        );
    }

    if (!row[sql::col::AGE_TYPES].is_null()) {
        preset.entries.ageTypes = common::parseSqlArray<AgeType>(
            row[sql::col::AGE_TYPES].as<std::string>()
        );
    }

    if (!row[sql::col::HIDDEN].is_null()) {
        preset.entries.hidden = row[sql::col::HIDDEN].as<bool>();
    }

    if (!row[sql::col::STATUS].is_null()) {
        preset.entries.status = boost::lexical_cast<UIFilterStatus>(
            row[sql::col::STATUS].as<std::string>());
    }

    return preset;
}

std::map<std::string, std::string>
createColumnToValue(
    const std::string& presetName,
    const PresetRoles& roles,
    const PresetEntries& entries,
    pqxx::transaction_base& txn)
{
    std::map<std::string, std::string> map;

    map.emplace(
        sql::col::NAME,
        txn.quote(presetName)
    );

    for (auto kind : enum_io::enumerateValues<RoleKind>()) {
        auto roleId = roles.getId(kind);
        if (roleId) {
            map.emplace(
                ROLE_KIND_TO_COL_NAME.at(kind),
                txn.quote(roleId)
            );
        }
    }

    map.emplace(
        sql::col::TYPES,
        toSqlArrayOrNull(entries.types, txn)
    );

    map.emplace(
        sql::col::WORKFLOWS,
        toSqlArrayOrNull(entries.workflows, txn)
    );

    map.emplace(
        sql::col::SOURCES,
        toSqlJsonArrayOrNull(entries.sources, txn)
    );

    map.emplace(
        sql::col::AGE_TYPES,
        toSqlArrayOrNull(entries.ageTypes, txn)
    );

    map.emplace(
        sql::col::HIDDEN,
        entries.hidden.has_value() ? toPgValue(entries.hidden.value()) : "NULL"
    );

    map.emplace(
        sql::col::STATUS,
        entries.status.has_value() ? toQuotedString(txn, entries.status.value()) : "NULL"
    );

    return map;
}

} // unnamed namespace

DEFINE_ENUM_IO(RoleKind, ROLE_KIND_STRINGS);

void PresetRoles::setId(RoleKind kind, TId id)
{
    roleSet_[kind] = id;
}

TId PresetRoles::getId(RoleKind kind) const
{
    auto it = roleSet_.find(kind);
    if (it == roleSet_.end()) {
        return 0;
    }
    return it->second;
}

bool PresetRoles::hasAllRoles() const
{
    return roleSet_.size() == enum_io::enumerateValues<RoleKind>().size();
}

BaseDimensions Preset::getBaseDimensions() const
{
    return BaseDimensions()
        .workflows(entries.workflows)
        .types(entries.types)
        .sources(entries.sources)
        .hidden(entries.hidden);
}

std::vector<Preset> getPresets(
    pqxx::transaction_base& txn)
{
    std::stringstream query;
    query << "SELECT " << PRESET_FIELDS
          << " FROM " << sql::table::FEEDBACK_PRESET
          << " ORDER BY " << sql::col::NAME;

    Presets presets;
    for (const auto& row : txn.exec(query)) {
        presets.push_back(presetFromPqxxRow(row));
    }
    return presets;
}

std::optional<Preset> getPreset(
    pqxx::transaction_base& txn,
    TId presetId)
{
    std::stringstream query;
    query << "SELECT " << PRESET_FIELDS
          << " FROM " << sql::table::FEEDBACK_PRESET
          << " WHERE " << sql::col::ID << " = " << presetId;

    const auto& rows = txn.exec(query);
    if (rows.empty()) {
        return std::nullopt;
    }
    ASSERT(rows.size() == 1);
    return presetFromPqxxRow(rows[0]);
}

Preset addPreset(
    pqxx::transaction_base& txn,
    const NewPreset& newPreset)
{
    REQUIRE(
        newPreset.roles.hasAllRoles(),
        CreateNoRolePresetError() << "Cannot create preset without all related roles"
    );
    auto columnToValue = createColumnToValue(
        newPreset.name,
        newPreset.roles,
        newPreset.entries,
        txn
    );

    std::stringstream query;
    query
        << "INSERT INTO " << sql::table::FEEDBACK_PRESET << " "
        << "(" << common::join(boost::adaptors::keys(columnToValue), ',') << ")"
        << " VALUES "
        << "(" << common::join(boost::adaptors::values(columnToValue), ',') << ")"
        << " RETURNING " << PRESET_FIELDS;

    try {
        auto rows = txn.exec(query.str());
        ASSERT(!rows.empty());
        return presetFromPqxxRow(rows[0]);
    } catch (const pqxx::unique_violation&) {
        throw UniquePresetNameError()
            << "Preset with name " << newPreset.name << " already exists";
    }
}

void updatePreset(
    pqxx::transaction_base& txn,
    const Preset& preset)
{
    for (auto kind : enum_io::enumerateValues<RoleKind>()) {
        REQUIRE(
            preset.roles.getId(kind) == 0,
            UpdatePresetRoleError() << "Cannot update roles for preset"
        );
    }
    auto columnToValue = createColumnToValue(
        preset.name,
        PresetRoles{},
        preset.entries,
        txn
    );

    auto setClause = common::join(
        columnToValue,
        [](const auto& colVal) {
            return colVal.first + " = " + colVal.second;
        },
        ", "
    );

    std::stringstream query;
    query
        << "UPDATE " << sql::table::FEEDBACK_PRESET << " "
        << "SET " << setClause << " "
        << "WHERE " << sql::col::ID << " = " << preset.id << " "
        << "RETURNING " << PRESET_FIELDS;

    try {
        auto rows = txn.exec(query.str());
        if (rows.empty()) {
            throw PresetDoesntExistError()
                << "Preset with id " << preset.id << " doesnt exist";
        }
    } catch (const pqxx::unique_violation&) {
        throw UniquePresetNameError()
            << "Preset with name " << preset.name << " already exists";
    }
}

void deletePreset(
    pqxx::transaction_base& txn,
    uint64_t presetId)
{
    std::stringstream query;
    query
        << "DELETE FROM " << sql::table::FEEDBACK_PRESET << " "
        << "WHERE " << sql::col::ID << " = " << presetId;

    txn.exec(query.str());
}

} // namespace maps::wiki::social::feedback
