#include <yandex/maps/wiki/social/feedback/task_brief.h>

#include <maps/wikimap/mapspro/libs/social/feedback/util.h>
#include <maps/wikimap/mapspro/libs/social/feedback/workflow_logic.h>
#include <maps/wikimap/mapspro/libs/social/helpers.h>
#include <maps/wikimap/mapspro/libs/social/magic_strings.h>

#include <maps/libs/json/include/value.h>
#include <yandex/maps/wiki/social/feedback/task_patch.h>
#include <yandex/maps/wiki/social/feedback/consts.h>

namespace maps::wiki::social::feedback {

using boost::lexical_cast;

namespace {

geolib3::Point2
wkbStringToPoint(const std::string& wkbStr)
{
    return geolib3::WKB::read<geolib3::Point2>(wkbStr);
}

TUids parseTUidArray(const std::string& sqlArrayString)
{
    auto stringSet = parseStringSetFromSqlArray(sqlArrayString);
    TUids retVal;
    for (const auto& str : stringSet) {
        retVal.emplace(lexical_cast<TUid>(str));
    }
    return retVal;
}

TaskState taskState(
    Bucket bucket,
    const std::optional<chrono::TimePoint>& deployedAt,
    const std::optional<Resolution>& resolution)
{
    if (bucket == Bucket::Incoming) {
        return TaskState::Incoming;
    }
    if (bucket == Bucket::Deferred) {
        return TaskState::Deferred;
    }
    if (bucket == Bucket::NeedInfo) {
        return TaskState::NeedInfo;
    }
    if (deployedAt) {
        return TaskState::Deployed;
    }
    if (resolution) {
        if (resolution->verdict() == Verdict::Accepted) {
            return TaskState::Accepted;
        } else {
            return TaskState::Rejected;
        }
    }
    return TaskState::Opened;
}

} // namespace anonymous

TaskState TaskBrief::state(const TaskPatch& patch) const
{
    auto bucket = patch.bucket()
                  ? patch.bucket().value()
                  : this->bucket();
    auto deployedAt = patch.deployedAt()
                      ? patch.deployedAt()
                      : this->deployedAt();
    auto resolution = [&]() -> std::optional<Resolution> {
        if (patch.resolution()) {
            return patch.resolution().value();
        } else if (resolved()) {
            return resolved()->resolution;
        } else {
            return std::nullopt;
        }
    }();

    return taskState(
        bucket,
        deployedAt,
        resolution
    );
}

TaskState TaskBrief::state() const
{
    static const TaskPatch emptyPatch(0);
    return state(emptyPatch);
}

TaskBrief::TaskBrief(const pqxx::row& row, const std::string& tablePrefix) :

    id_(rowField(row, sql::col::ID, tablePrefix).as<TId>()),

    createdAt_(chrono::parseSqlDateTime(
        rowField(row, sql::col::CREATED_AT, tablePrefix).as<std::string>())),

    position_(wkbStringToPoint(
        pqxx::binarystring(rowField(row, sql::col::POSITION, tablePrefix)).str())),

    indoorLevel_(rowField(row, sql::col::INDOOR_LEVEL, tablePrefix).as<std::string>()),

    type_(lexical_cast<Type>(
        rowField(row, sql::col::TYPE, tablePrefix).as<std::string>())),

    bucket_(lexical_cast<Bucket>(
        rowField(row, sql::col::BUCKET, tablePrefix).as<std::string>())),

    source_(rowField(row, sql::col::SOURCE, tablePrefix).as<std::string>()),

    hidden_(rowField(row, sql::col::HIDDEN, tablePrefix).as<bool>()),

    internalContent_(rowField(row, sql::col::INTERNAL_CONTENT, tablePrefix).as<bool>()),

    hashValue_(rowField(row, sql::col::HASH_VALUE, tablePrefix).as<std::string>()),

    stateModifiedAt_(chrono::parseSqlDateTime(
        rowField(row, sql::col::STATE_MODIFIED_AT, tablePrefix).as<std::string>())),

    processingLvl_(enum_io::fromString<ProcessingLvl>(
        rowField(row, sql::col::PROCESSING_LVL, tablePrefix).as<std::string>()))
{
    if (!rowField(row, sql::col::VIEWED_BY, tablePrefix).is_null()) {
        viewedBy_ = parseTUidArray(
            rowField(row, sql::col::VIEWED_BY, tablePrefix).as<std::string>());
    }

    if (!rowField(row, sql::col::ACQUIRED_BY, tablePrefix).is_null()) {
        acquired_ = TaskAcquired {
            rowField(row, sql::col::ACQUIRED_BY, tablePrefix).as<TUid>(),
            chrono::parseSqlDateTime(
                rowField(row, sql::col::ACQUIRED_AT, tablePrefix).as<std::string>())
        };
    }

    if (!rowField(row, sql::col::RESOLVED_BY, tablePrefix).is_null()) {
        std::optional<RejectReason> rejectReason;

        if (!rowField(row, sql::col::REJECT_REASON, tablePrefix).is_null()) {
            rejectReason = lexical_cast<RejectReason>(
                rowField(row, sql::col::REJECT_REASON, tablePrefix)
                    .as<std::string>());
        }

        resolved_ = TaskResolved{
            rowField(row, sql::col::RESOLVED_BY, tablePrefix).as<TUid>(),
            chrono::parseSqlDateTime(
                rowField(row, sql::col::RESOLVED_AT, tablePrefix).as<std::string>()),
            Resolution(
                lexical_cast<Verdict>(
                    rowField(row, sql::col::RESOLUTION, tablePrefix).as<std::string>()),
                rejectReason)
        };
    }

    if (!rowField(row, sql::col::DEPLOYED_AT, tablePrefix).is_null()) {
        deployedAt_ = chrono::parseSqlDateTime(
            rowField(row, sql::col::DEPLOYED_AT, tablePrefix).as<std::string>());
    }

    if (!rowField(row, sql::col::OBJECT_ID, tablePrefix).is_null()) {
        objectId_ = rowField(row, sql::col::OBJECT_ID, tablePrefix).as<TId>();
        REQUIRE(
            objectId_.value() != 0,
            "Object is is 0 for feedback task " << id_
        );
    }

    if (!rowField(row, sql::col::DUPLICATE_HEAD_ID, tablePrefix).is_null()) {
        duplicateHeadId_ = rowField(row, sql::col::DUPLICATE_HEAD_ID, tablePrefix).as<TId>();
    }
}

TId TaskBrief::id() const
{
    return id_;
}

chrono::TimePoint TaskBrief::createdAt() const
{
    return createdAt_;
}

chrono::TimePoint TaskBrief::stateModifiedAt() const
{
    return stateModifiedAt_;
}

int TaskBrief::processingLevel() const
{
    return processingLvl_ == ProcessingLvl::Level0 ? 0 : 1;
}

ProcessingLvl TaskBrief::processingLvl() const
{
    return processingLvl_;
}

const geolib3::Point2& TaskBrief::position() const
{
    return position_;
}

const std::string& TaskBrief::indoorLevel() const
{
    return indoorLevel_;
}

Type TaskBrief::type() const
{
    return type_;
}

Workflow TaskBrief::workflow() const
{
    return feedback::workflow(*this);
}

Bucket TaskBrief::bucket() const
{
    return bucket_;
}

const std::string& TaskBrief::source() const
{
    return source_;
}

bool TaskBrief::hidden() const
{
    return hidden_;
}

bool TaskBrief::internalContent() const
{
    return internalContent_;
}

const std::optional<TaskAcquired>& TaskBrief::acquired() const
{
    return acquired_;
}

const std::optional<TaskResolved>& TaskBrief::resolved() const
{
    return resolved_;
}

const std::optional<chrono::TimePoint>& TaskBrief::deployedAt() const
{
    return deployedAt_;
}

const std::optional<TId>& TaskBrief::objectId() const
{
    return objectId_;
}

std::string TaskBrief::hashValue() const
{
    return hashValue_;
}

const std::optional<TId>& TaskBrief::duplicateHeadId() const
{
    return duplicateHeadId_;
}

bool TaskBrief::revealed() const
{
    return bucket_ != Bucket::Incoming
        && bucket_ != Bucket::Deferred;
}

const TUids& TaskBrief::viewedBy() const
{
    return viewedBy_;
}

AgeType TaskBrief::ageType() const
{
    if (workflow() == Workflow::Task) {
        return AgeType::New;
    }

    bool isOldTask =
        chrono::TimePoint::clock::now() - createdAt_ >=
            OLD_FEEDBACK_TASK_AGE_HOURS;

    return isOldTask ? AgeType::Old : AgeType::New;
}

Resolution Resolution::createAccepted()
{
    return Resolution(Verdict::Accepted, std::nullopt);
}

Resolution Resolution::createRejected(std::optional<RejectReason> rejectReason)
{
    return Resolution(Verdict::Rejected, rejectReason);
}

Resolution::Resolution(
    Verdict verdict, std::optional<RejectReason> rejectReason)
    : verdict_(verdict), rejectReason_(rejectReason)
{ }

bool Resolution::operator==(const Resolution& other) const
{
    return (verdict_ == other.verdict_) &&
        (rejectReason_ == other.rejectReason_);
}

} // namespace maps::wiki::social::feedback
