#include <yandex/maps/wiki/social/feedback/task_filter.h>

#include <maps/wikimap/mapspro/libs/social/feedback/util.h>
#include <maps/wikimap/mapspro/libs/social/feedback/workflow_logic.h>
#include <maps/wikimap/mapspro/libs/social/helpers.h>

#include <maps/libs/common/include/exception.h>
#include <yandex/maps/wiki/common/pg_utils.h>

#include <fmt/format.h>
#include <algorithm>

using namespace fmt::literals;

namespace maps::wiki::social::feedback {

namespace {

const std::string LAST_NEED_INFO_SUBQUERY =
    "EXISTS"
    "("
        "SELECT "
            "1 "
        "FROM "
            "("
                " SELECT "
                    + sql::col::MODIFIED_AT + ","
                    + sql::col::MODIFIED_BY +
                " FROM "
                    + sql::table::FEEDBACK_HISTORY +
                " WHERE "
                    + sql::col::FEEDBACK_TASK_ID + " = " + sql::col::ID +
                    " AND " + sql::col::OPERATION + " = '"
                        + std::string(toString(TaskOperation::NeedInfo)) + "'"
                " ORDER BY "
                    + sql::col::MODIFIED_AT + " DESC "
                " LIMIT 1"
            ") as last_need_info "
        "WHERE "
            "{condition}"
    ")";

std::string lastNeedInfoFilterExpr(
    pqxx::transaction_base& txn,
    const std::optional<DateTimeCondition>& at,
    const std::optional<TUid>& by)
{
    std::stringstream condition;
    condition << "TRUE";

    if (at) {
         condition << " AND " << at->sqlComparison(txn, sql::col::MODIFIED_AT);
    }
    if (by) {
        condition << " AND " << sql::col::MODIFIED_BY << " = " << *by;
    }

    return fmt::format(LAST_NEED_INFO_SUBQUERY, "condition"_a = condition.str());
}

std::string ageTypesFilterExpr(AgeTypes ageTypes)
{
    if (ageTypes.empty()) {
        return "FALSE";
    }

    std::sort(ageTypes.begin(), ageTypes.end());
    ageTypes.erase(
        std::unique(ageTypes.begin(), ageTypes.end()),
        ageTypes.end()
    );

    // Check if all types are present
    //
    auto all = allAgeTypes();
    std::sort(all.begin(), all.end());
    if (all == ageTypes) {
        return "TRUE";
    }

    // Case of single age type
    //
    ASSERT(ageTypes.size() == 1);
    return ageTypeSqlCondition(ageTypes.front());
}

std::string isNotNullCondition(bool pred)
{
    return (pred ? "IS NOT " : "IS ") + sql::value::NULL_;
}

const TaskOperations CONSIDERED_MODIFICATION_OPERATIONS {
    TaskOperation::Accept,
    TaskOperation::Reject,
    TaskOperation::NeedInfo,
    TaskOperation::ProcessingLevelUp,
    TaskOperation::ProcessingLevelDown,
    TaskOperation::ChangePosition,
    TaskOperation::ChangeType,
    TaskOperation::ChangeProcessingLvl,
};

} // namespace anonymous

std::string BaseDimensions::whereClause() const
{
    std::stringstream query;
    query << "TRUE";

    if (types_) {
        query << " AND " << sql::col::TYPE << " "
            << common::sqlInCondition(*types_);
    }
    if (workflows_) {
        query << " AND " << feedback::whereClause(*workflows_);
    }
    if (sources_) {
        query << " AND " << sql::col::SOURCE << " "
            << common::sqlInCondition(*sources_);
    }
    if (hidden_) {
        query << " AND " << sql::col::HIDDEN << " = " << toPgValue(*hidden_);
    }

    return query.str();
}

BaseDimensions& BaseDimensions::hidden(std::optional<bool> hidden)
{
    hidden_ = std::move(hidden);
    return *this;
}

BaseDimensions& BaseDimensions::types(std::optional<Types> types)
{
    types_ = std::move(types);
    return *this;
}

BaseDimensions& BaseDimensions::workflows(std::optional<Workflows> workflows)
{
    workflows_ = std::move(workflows);
    return *this;
}

BaseDimensions& BaseDimensions::sources(std::optional<std::vector<std::string>> sources)
{
    sources_ = std::move(sources);
    return *this;
}

const std::optional<Workflows>& BaseDimensions::getWorkflows() const
{
    return workflows_;
}
const std::optional<Types>& BaseDimensions::getTypes() const
{
    return types_;
}
const std::optional<std::vector<std::string>>& BaseDimensions::getSources() const
{
    return sources_;
}
const std::optional<bool>& BaseDimensions::getHidden() const
{
    return hidden_;
}

std::string TaskFilter::joinClause() const
{
    std::stringstream query;

    if (createdBy_ || modifiedBy_) {
        REQUIRE(!createdBy_ || !modifiedBy_ || *createdBy_ == *modifiedBy_,
            "Invalid taskFilter (createdBy != modifiedBy): " <<
                *createdBy_ << " != " << *modifiedBy_);

        TUid uid = 0;
        TaskOperations taskOperations;
        if (modifiedBy_) {
            uid = *modifiedBy_;
            taskOperations = CONSIDERED_MODIFICATION_OPERATIONS;
        }
        if (createdBy_) {
            uid = *createdBy_;
            taskOperations.emplace(TaskOperation::Create);
        }

        std::stringstream queryHistory;
        queryHistory
            << sql::col::MODIFIED_BY << " = " << uid
            << " AND " << sql::col::OPERATION << " " << common::sqlInCondition(taskOperations);

        if (!modifiedBy_) { // only createdBy
            query
                << " JOIN " << sql::table::FEEDBACK_HISTORY
                << " ON " << queryHistory.str()
                << " AND " << sql::col::ID << " = " << sql::col::FEEDBACK_TASK_ID;
        } else {
            query
                << " JOIN ("
                << "SELECT DISTINCT " << sql::col::FEEDBACK_TASK_ID
                << " FROM " << sql::table::FEEDBACK_HISTORY
                << " WHERE " << queryHistory.str()
                << ") fh ON " << sql::col::ID << " = " << sql::col::FEEDBACK_TASK_ID;
        }
    }

    static const std::string AOI_TABLE_ALIAS = "aoi";
    if (aoiId_) {
        query
            << " JOIN " << sql::table::FEEDBACK_AOI_FEED << " AS " << AOI_TABLE_ALIAS
            << " ON " << sql::col::ID << " = " << AOI_TABLE_ALIAS << "." << sql::col::FEEDBACK_TASK_ID;
    }

    return query.str();
}

std::string TaskFilter::whereClause(pqxx::transaction_base& txn) const
{
    std::stringstream query;

    query << "TRUE";

    if (ids_) {
        query << " AND " << sql::col::ID << " "
              << common::sqlInCondition(*ids_);
    }
    if (idGreaterThan_) {
        query << " AND " << sql::col::ID << " > " << *idGreaterThan_;
    }
    if (boundary_) {
        query << " AND " << sql::func::ST_WITHIN << "(" << sql::col::POSITION
              << ", " << makePqxxGeomExpr(txn, *boundary_) << ")";
    }
    if (multiPolygon_) {
        query << " AND " << sql::func::ST_WITHIN << "(" << sql::col::POSITION
              << ", " << makePqxxGeomExpr(txn, *multiPolygon_) << ")";
    }
    if (boundingBox_) {
        query << " AND " << sql::col::POSITION << " && "
              << makePqxxGeomExpr(txn, boundingBox_->polygon());
    }
    if (indoorLevel_) {
        query << " AND " << sql::col::INDOOR_LEVEL
              << " IN ('', " << txn.quote(*indoorLevel_) << ")";
    }
    if (duplicateHeadIds_) {
        query << " AND " << sql::col::DUPLICATE_HEAD_ID << " "
              << common::sqlInCondition(*duplicateHeadIds_);
    }
    if (acquiredBy_) {
        query << " AND " << sql::col::ACQUIRED_BY << " = " << *acquiredBy_;
    }
    if (resolvedBy_) {
        query << " AND " << sql::col::RESOLVED_BY << " = " << *resolvedBy_;
    }
    if (lastNeedInfoBy_ || lastNeedInfoAt_) {
        query << " AND " << lastNeedInfoFilterExpr(txn, lastNeedInfoAt_, lastNeedInfoBy_);
    }
    if (notResolvedBy_) {
        query << " AND "
            << "("
            << sql::col::RESOLVED_BY << " IS NULL"
            << " OR "
            << " NOT " << sql::col::RESOLVED_BY << " " << common::sqlInCondition(*notResolvedBy_)
            << ")";
    }
    if (resolvedAt_) {
        query << " AND " << resolvedAt_->sqlComparison(txn, sql::col::RESOLVED_AT);
    }
    if (objectId_) {
        query << " AND " << sql::col::OBJECT_ID << " = " << *objectId_;
    }
    if (verdict_) {
        query << " AND " << sql::col::RESOLUTION
              << " = '" << *verdict_ << "'";
    }
    if (resolved_) {
        query << " AND " << sql::col::RESOLVED_BY << " "
              << isNotNullCondition(*resolved_);
    }
    if (deployed_) {
        query << " AND " << sql::col::DEPLOYED_AT << " "
              << isNotNullCondition(*deployed_);
    }
    if (hidden_) {
        query << " AND " << sql::col::HIDDEN << " = " << toPgValue(*hidden_);
    }
    if (internalContent_) {
        query << " AND " << sql::col::INTERNAL_CONTENT << " = " << toPgValue(*internalContent_);
    }

    if (types_) {
        query << " AND " << sql::col::TYPE << " "
              << common::sqlInCondition(*types_);
    }

    if (workflows_) {
        query << " AND " << feedback::whereClause(*workflows_);
    }

    if (sources_) {
        query << " AND " << sql::col::SOURCE << " "
              << common::sqlInCondition(*sources_);
    }

    if (notSources_) {
        query << " AND " << sql::col::SOURCE << " NOT "
              << common::sqlInCondition(*notSources_);
    }

    if (isExperiment_) {
        query << " AND " << sql::col::SOURCE
              << (*isExperiment_ ? "" : " NOT")
              << " LIKE '" << FEEDBACK_SOURCE_EXPERIMENT_PREFIX << "%'";
    }

    if (ageTypes_) {
        query << " AND " << ageTypesFilterExpr(*ageTypes_);
    }

    if (buckets_) {
        query << " AND " << sql::col::BUCKET << " "
            << common::sqlInCondition(*buckets_);
    }

    if (stateModifiedAt_) {
        query << " AND " << stateModifiedAt_->sqlComparison(txn, sql::col::STATE_MODIFIED_AT);
    }

    if (createdAt_) {
        query << " AND " << createdAt_->sqlComparison(txn, sql::col::CREATED_AT);
    }

    if (createdBeforeNow_) {
        query << " AND " << sql::col::CREATED_AT << " < " << sql::value::NOW;
    }

    if (processingLvls_) {
        query << " AND " << sql::col::PROCESSING_LVL << " "
            << common::sqlInCondition(*processingLvls_);
    }

    if (aoiId_) {
        query << " AND " << sql::col::AOI_ID << " = " << *aoiId_;
    }
    if (!baseDimensionsArray_.empty()) {
        query << internal::BaseDimensionsArrayWhereClause(baseDimensionsArray_);
    }

    return query.str();
}

TaskFilter& TaskFilter::ids(TIds ids)
{
    ids_ = std::move(ids);
    return *this;
}

TaskFilter& TaskFilter::idGreaterThan(TId id)
{
    idGreaterThan_ = id;
    return *this;
}

TaskFilter& TaskFilter::boundary(const geolib3::Polygon2& boundary)
{
    boundary_ = boundary;
    return *this;
}

TaskFilter& TaskFilter::boundary(const geolib3::MultiPolygon2& multiPolygon)
{
    multiPolygon_ = multiPolygon;
    return *this;
}

TaskFilter& TaskFilter::boxBoundary(const geolib3::BoundingBox& boundingBox)
{
    boundingBox_ = boundingBox;
    return *this;
}

TaskFilter& TaskFilter::indoorLevel(std::optional<std::string> indoorLevel)
{
    indoorLevel_ = std::move(indoorLevel);
    return *this;
}

TaskFilter& TaskFilter::acquiredBy(TUid acquiredBy)
{
    acquiredBy_ = acquiredBy;
    return *this;
}

TaskFilter& TaskFilter::resolvedBy(std::optional<TUid> resolvedBy)
{
    resolvedBy_ = resolvedBy;
    return *this;
}

TaskFilter& TaskFilter::lastNeedInfoBy(std::optional<TUid> lastNeedInfoBy)
{
    lastNeedInfoBy_ = lastNeedInfoBy;
    return *this;
}

TaskFilter& TaskFilter::modifiedBy(TUid modifiedBy)
{
    modifiedBy_ = modifiedBy;
    return *this;
}

TaskFilter& TaskFilter::notResolvedBy(TUids notResolvedBy)
{
    notResolvedBy_ = std::move(notResolvedBy);
    return *this;
}

TaskFilter& TaskFilter::resolvedAt(std::optional<DateTimeCondition> condition)
{
    resolvedAt_ = std::move(condition);
    return *this;
}

TaskFilter& TaskFilter::lastNeedInfoAt(std::optional<DateTimeCondition> condition)
{
    lastNeedInfoAt_ = std::move(condition);
    return *this;
}

TaskFilter& TaskFilter::verdict(std::optional<Verdict> verdict)
{
    verdict_ = verdict;
    return *this;
}

TaskFilter& TaskFilter::resolved(bool resolved)
{
    resolved_ = resolved;
    return *this;
}

TaskFilter& TaskFilter::deployed(bool deployed)
{
    deployed_ = deployed;
    return *this;
}

TaskFilter& TaskFilter::hidden(std::optional<bool> hidden)
{
    hidden_ = hidden;
    return *this;
}

TaskFilter& TaskFilter::internalContent(bool internalContent)
{
    internalContent_ = internalContent;
    return *this;
}

TaskFilter& TaskFilter::type(Type type)
{
    types_ = Types{type};
    return *this;
}

TaskFilter& TaskFilter::types(std::optional<Types> types)
{
    types_ = std::move(types);
    return *this;
}

TaskFilter& TaskFilter::processingLvls(std::optional<ProcessingLvlSet> value)
{
    processingLvls_ = std::move(value);
    return *this;
}

TaskFilter& TaskFilter::ageType(AgeType ageType)
{
    ageTypes_ = AgeTypes{ageType};
    return *this;
}

TaskFilter& TaskFilter::ageTypes(std::optional<AgeTypes> ageTypes)
{
    ageTypes_ = std::move(ageTypes);
    return *this;
}

TaskFilter& TaskFilter::workflow(Workflow workflow)
{
    workflows_ = Workflows{workflow};
    return *this;
}

TaskFilter& TaskFilter::workflows(std::optional<Workflows> workflows)
{
    workflows_ = std::move(workflows);
    return *this;
}

TaskFilter& TaskFilter::bucket(Bucket bucket)
{
    return buckets({bucket});
}

TaskFilter& TaskFilter::buckets(Buckets buckets)
{
    buckets_ = std::move(buckets);
    return *this;
}

TaskFilter& TaskFilter::duplicateHeadId(std::optional<TId> headId)
{
    if (headId) {
        ASSERT(*headId);
        duplicateHeadIds_ = TIds{*headId};
    } else {
        duplicateHeadIds_ = TIds{};
    }
    return *this;
}

TaskFilter& TaskFilter::objectId(TId objectId)
{
    objectId_ = objectId;
    return *this;
}

TaskFilter& TaskFilter::aoiId(TId aoiId)
{
    aoiId_ = aoiId;
    return *this;
}

TaskFilter& TaskFilter::duplicateHeadIds(TIds headIds)
{
    duplicateHeadIds_ = std::move(headIds);
    return *this;
}

TaskFilter& TaskFilter::source(std::string source)
{
    sources_ = std::vector<std::string>{std::move(source)};
    return *this;
}

TaskFilter& TaskFilter::sourceNotIn(std::vector<std::string> notSources)
{
    REQUIRE(
        !notSources.empty(),
        LogicError("Empty list of sources is passed to filter.")
    );

    notSources_ = std::move(notSources);
    return *this;
}

TaskFilter& TaskFilter::sources(std::optional<std::vector<std::string>> sources)
{
    sources_ = std::move(sources);
    return *this;
}

TaskFilter& TaskFilter::experiment(bool isExperiment)
{
    isExperiment_ = isExperiment;
    return *this;
}

TaskFilter& TaskFilter::stateModifiedAt(DateTimeCondition condition)
{
    stateModifiedAt_ = std::move(condition);
    return *this;
}

TaskFilter& TaskFilter::createdBy(TUid createdBy)
{
    createdBy_ = createdBy;
    return *this;
}

TaskFilter& TaskFilter::createdAt(DateTimeCondition condition)
{
    createdAt_ = std::move(condition);
    return *this;
}

TaskFilter& TaskFilter::createdBeforeNow()
{
    createdBeforeNow_ = true;
    return *this;
}

TaskFilter& TaskFilter::addBaseDimensions(BaseDimensions baseDimensions)
{
    baseDimensionsArray_.emplace_back(std::move(baseDimensions));
    return *this;
}

TaskFilter& TaskFilter::uiFilterStatus(std::optional<UIFilterStatus> uiFilterStatus)
{
    if (!uiFilterStatus) {
        resolved_ = std::nullopt;
        buckets(allRevealedBuckets());
        return *this;
    }
    switch (*uiFilterStatus) {
        case UIFilterStatus::Opened:
            resolved(false);
            bucket(Bucket::Outgoing);
            break;

        case UIFilterStatus::NeedInfo:
            resolved(false);
            bucket(Bucket::NeedInfo);
            break;

        case UIFilterStatus::Resolved:
            resolved(true);
            bucket(Bucket::Outgoing);
            break;
    }
    return *this;
}

namespace internal {

std::string BaseDimensionsArrayWhereClause(const std::vector<BaseDimensions>& baseDimensionsArray)
{
    std::ostringstream clause;
    clause << " AND ( FALSE ";
    for (const auto& baseDimensions : baseDimensionsArray) {
        clause << " OR (" << baseDimensions.whereClause() << ")";
    }
    clause << ")";
    return clause.str();
}

} // namespace internal

} // namespace maps::wiki::social::feedback
